jQuery(document).ready(function($) {
    // Store password in memory (not in localStorage for security)
    let storedPassword = '';
    // Add an interval reference variable
    let articlesRefreshInterval;

    // Check if we have a stored password in the hidden input
    const $passwordInput = $('#autoarticle-password');
    if ($passwordInput.length) {
        storedPassword = $passwordInput.val();
    }

    // Function to show loading overlay
    function showLoadingOverlay(message = 'Saving...') {
        const $overlay = $('.autoarticle-loading-overlay');
        $overlay.find('.autoarticle-loading-text').text(message);
        $overlay.show();
    }

    // Function to hide loading overlay
    function hideLoadingOverlay() {
        $('.autoarticle-loading-overlay').hide();
    }

    function showMessage(message, type, context = 'login') {
        console.log('Showing message:', message, type, context);
        const $message = context === 'login' ? $('#login-message') : $('#generation-message');
        if (!$message.length) {
            console.error('Message element not found:', context);
            return;
        }
        
        // Only show message if there's content
        if (message) {
            $message
                .removeClass('notice-success notice-error notice-warning notice-info')
                .addClass('notice-' + (type === 'success' ? 'success' : 'error'))
                .html('<p>' + message + '</p>')
                .show();
            
            // Scroll to the top of the screen to make the message visible
            if (context === 'generation') {
                $('html, body').animate({ scrollTop: 0 }, 'fast');
            }
        } else {
            $message.hide();
        }
    }

    // Function to update membership UI based on level
    function updateMembershipUI(membershipLevel, membershipName) {
        // Update membership level display
        $('#membership-level-name').text(membershipName || 'Unknown');
        
        // Set up the upgrade button for non-Pro plans
        const $upgradeContainer = $('#upgrade-button-container');
        $upgradeContainer.empty();
        
        if (membershipLevel && membershipLevel < 3) {
            $upgradeContainer.html('<a href="https://autoarticle.net/membership-levels/" class="button   t-center w-100 mt-2" target="_blank">Upgrade</a>');
        }
        
        // Enable scheduling features for Premium (2) and Pro (3) plans
        const $frequency = $('#frequency');
        const $scheduleTime = $('#schedule_time');
        
        if (membershipLevel && membershipLevel >= 2) {
            $frequency.prop('disabled', false);
            $scheduleTime.prop('disabled', false);
        } else {
            $frequency.prop('disabled', true);
            $scheduleTime.prop('disabled', true);
        }
    }

    // Function to update articles remaining count
    function updateArticlesRemaining(count) {
        $('#articles-remaining').text(count);
    }

    // Function to get articles remaining from remote server
    function getArticlesRemaining() {
        // Check if user is logged in with credentials
        if (autoarticle.user_email && autoarticle.user_password) {
            // Logged in user
            $.ajax({
                url: autoarticle.api_url,
                method: 'GET',
                data: {
                    action: 'get_articles_remaining',
                    email: autoarticle.user_email,
                    password: autoarticle.user_password,
                    external: 1
                },
                success: function(response) {
                    if (response && response.success && response.articles_remaining !== undefined) {
                        updateArticlesRemaining(response.articles_remaining);
                    } else {
                        updateArticlesRemaining('Error');
                    }
                },
                error: function() {
                    updateArticlesRemaining('Error');
                }
            });
        } else {
            // Anonymous user - check IP-based limit
            $.ajax({
                url: autoarticle.api_url,
                method: 'GET',
                data: {
                    action: 'check_ip_limit',
                    external: 1
                },
                success: function(response) {
                    if (response && response.success && response.articles_remaining !== undefined) {
                        updateArticlesRemaining(response.articles_remaining);
                        // Also set membership info for anonymous users
                        updateMembershipUI(1, 'Free');
                    } else {
                        updateArticlesRemaining('Error');
                        updateMembershipUI(1, 'Free');
                    }
                },
                error: function() {
                    updateArticlesRemaining('Error');
                    updateMembershipUI(1, 'Free');
                }
            });
        }
    }

    // Function to get membership status for a logged-in WordPress user (on the main site)
    function getMembershipStatusForWPUser() {
        // This function is intended to be called from the shortcode context on autoarticle.net
        // It makes a local AJAX call to a WordPress handler which can check the current user's level
        $.ajax({
            url: autoarticle.ajax_url, // Local WordPress AJAX endpoint
            method: 'POST',
            data: {
                action: 'autoarticle_get_current_user_membership', // A new AJAX action to be created
                nonce: autoarticle.nonce
            },
            success: function(response) {
                if (response.success) {
                    updateMembershipUI(response.data.membership_level, response.data.membership_name);
                    updateArticlesRemaining(response.data.articles_remaining);
                } else {
                    // Fallback to anonymous check if the local check fails for some reason
                    updateMembershipUI(1, 'Free');
                    updateArticlesRemaining(0);
                }
            },
            error: function() {
                // Fallback to anonymous check on AJAX error
                updateMembershipUI(1, 'Free');
                updateArticlesRemaining(0);
            }
        });
    }

    // Get articles remaining on page load for all users
    // getArticlesRemaining(); // REMOVED to prevent race condition. checkMembershipStatus now handles this.

    // Set up periodic refresh of articles remaining (every 30 seconds)
    // startArticlesRefreshInterval(); // Commented out to prevent periodic refresh

    function startArticlesRefreshInterval() {
        // Clear any existing interval
        clearInterval(articlesRefreshInterval);
        // Set up a new interval to refresh articles count every 30 seconds
        // articlesRefreshInterval = setInterval(getArticlesRemaining, 30000); // Commented out to prevent periodic refresh
    }

    // Modify the checkMembershipStatus function to update UI with membership info
    function checkMembershipStatus() {
        // If we are on the main site (identified by presence of hub_id in localized data), use the new WP User check
        if (typeof autoarticle.hub_id !== 'undefined') {
            getMembershipStatusForWPUser();
            return; // Exit the function to avoid running the old logic
        }

        if (autoarticle.user_email && storedPassword) {
            $.ajax({
                url: autoarticle.api_url,
                type: 'GET',
                data: {
                    external: 1,
                    action: 'verify_membership',
                    email: autoarticle.user_email,
                    password: storedPassword
                },
                success: function(response) {
                    if (response && response.success) {
                        updateArticlesRemaining(response.articles_remaining);
                        updateMembershipUI(response.membership_level, response.membership_name);
                    } else {
                        updateArticlesRemaining('Error');
                        updateMembershipUI(1, 'Free'); // Default to Free tier on error
                    }
                },
                error: function() {
                    updateArticlesRemaining('Error');
                    updateMembershipUI(1, 'Free'); // Default to Free tier on error
                }
            });
        } else {
            // Anonymous user - check IP-based limit
            getArticlesRemaining();
            updateMembershipUI(1, 'Free'); // Default to Free tier for anonymous users
        }
    }

    // Call checkMembershipStatus on page load
    checkMembershipStatus();
    
    // Handle refresh button click
    $(document).on('click', '#refresh-membership-status', function() {
        const $btn = $(this);
        const originalText = $btn.text();
        $btn.text('Refreshing...').prop('disabled', true);

        // Decide which check to run
        if (typeof autoarticle.hub_id !== 'undefined') {
            getMembershipStatusForWPUser();
        } else {
            checkMembershipStatus();
        }

        // Reset button text after a short delay
        setTimeout(() => {
            $btn.text(originalText).prop('disabled', false);
        }, 1500);
    });

    // Show login modal when the Login button is clicked
    $('#autoarticle-login-btn').on('click', function() {
        $('#autoarticle-login-modal').show();
    });
    
    // Close login modal when Cancel button is clicked
    $('#cancel-login, .autoarticle-modal-close').on('click', function() {
        $('#autoarticle-login-modal').hide();
    });

    // Add direct close handlers to ensure modals can be closed
    $(document).on('click', '#autoarticle-login-modal .autoarticle-modal-close', function() {
        $('#autoarticle-login-modal').hide();
    });
    
    $(document).on('click', '#cancel-login', function() {
        $('#autoarticle-login-modal').hide();
    });

    // Handle login form submission
    $('#autoarticle-login').on('submit', function(e) {
        e.preventDefault();
        const $form = $(this);
        const $submitBtn = $form.find('button[type="submit"]');
        
        // Get form data
        const email = $('#email').val();
        const password = $('#password').val();
        
        // Store password for future use
        storedPassword = password;
        
        // Create or update hidden input for password
        if (!$passwordInput.length) {
            $('<input>').attr({
                type: 'hidden',
                id: 'autoarticle-password',
                value: password
            }).appendTo('body');
        } else {
            $passwordInput.val(password);
        }
        
        // Disable submit button and show loading state
        $submitBtn.prop('disabled', true).addClass('loading');
        $form.css('opacity', '0.5');
        
        // Make AJAX request to prose.digital
        $.ajax({
            url: autoarticle.api_url,
            type: 'POST',
            data: {
                email: email,
                password: password
            },
            success: function(response) {
                console.log('Login response:', response);
                if (response.success) {
                    // Store credentials but not membership level
                    $.ajax({
                        url: autoarticle.ajax_url,
                        type: 'POST',
                        data: {
                            action: 'autoarticle_save_membership',
                            nonce: autoarticle.nonce,
                            email: email,
                            password: password
                        },
                        success: function() {
                            // Update UI
                            $('#autoarticle-login-modal').hide();
                            $('#autoarticle-login-btn').hide();
                            $('#autoarticle-logout').show();
                            $('#autoarticle-trigger-overdue').show();
                            
                            // Update membership info
                            updateMembershipUI(response.membership_level, response.membership_name);
                            updateArticlesRemaining(response.articles_remaining);
                            
                            // Show success message
                            showMessage('Successfully logged in!', 'success', 'login');
                        },
                        error: function() {
                            showMessage('Error saving credentials. Please try again.', 'error', 'login');
                        }
                    });
                } else {
                    showMessage('Invalid email or password. Please try again.', 'error', 'login');
                }
            },
            error: function() {
                showMessage('Error connecting to server. Please try again.', 'error', 'login');
            },
            complete: function() {
                // Re-enable submit button and remove loading state
                $submitBtn.prop('disabled', false).removeClass('loading');
                $form.css('opacity', '1');
            }
        });
    });

    // Handle logout button click
    $('#autoarticle-logout').on('click', function() {
        const $button = $(this);
        $button.prop('disabled', true).addClass('loading');
        
        // Clear stored password
        storedPassword = '';
        if ($passwordInput.length) {
            $passwordInput.val('');
        }
        
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_logout',
                nonce: autoarticle.nonce
            },
            success: function(response) {
                if (response.success) {
                    showMessage('Logging out...', 'success');
                    setTimeout(function() {
                        location.reload();
                    }, 1000);
                } else {
                    showMessage('Failed to logout. Please try again.', 'error');
                    $button.prop('disabled', false).removeClass('loading');
                }
            },
            error: function() {
                showMessage('Failed to logout. Please try again.', 'error');
                $button.prop('disabled', false).removeClass('loading');
            }
        });
        
        // Clear the refresh interval when logging out
        clearInterval(articlesRefreshInterval);
    });

    // Handle article generation form submission
    $('#autoarticle-generate').off('submit').on('submit', function(e) {
        e.preventDefault();
        handleGeneration();
    });

    // Handle generate now button click
    $('#autoarticle-generate-now').off('click').on('click', function(e) {
        e.preventDefault();
        handleGeneration();
    });

    function handleGeneration() {
        const $form = $('#autoarticle-generate');
        const $submitBtn = $('#autoarticle-generate-now');
        
        // Get form data
        const topics = JSON.parse($('#topics').val() || '[]');
        const style = $('#style').val();
        const imageStyle = $('#image_style').val();
        
        // Always show topic selection modal, even if no topics are added
        showTopicSelectionModal(topics, style, imageStyle, $submitBtn, $form);
    }

    function showTopicSelectionModal(topics, style, imageStyle, $submitBtn, $form) {
        const $modal = $('#autoarticle-topic-modal');
        const $topicList = $modal.find('.autoarticle-topic-list');
        const $customTopic = $('#custom-topic');
        
        // Clear previous content
        $topicList.empty();
        $customTopic.val('');
        
        // Add existing topics to the list
        topics.forEach(topic => {
            $('<div>')
                .addClass('autoarticle-topic-item')
                .text(topic)
                .appendTo($topicList);
        });
        
        $modal.show();
        
        // Remove old handlers before adding new ones
        $('.autoarticle-topic-item').off('click');
        $('#confirm-topic').off('click');
        $('#cancel-topic').off('click');
        $('.autoarticle-modal-close').off('click');
        
        // Handle topic selection
        $('.autoarticle-topic-item').on('click', function() {
            $('.autoarticle-topic-item').removeClass('selected');
            $(this).addClass('selected');
            // Clear custom topic input when a predefined topic is selected
            $customTopic.val('');
        });
        
        // Handle custom topic input
        $customTopic.on('input', function() {
            // Deselect any selected predefined topic when typing in custom topic
            $('.autoarticle-topic-item').removeClass('selected');
        });
        
        // Handle confirm button
        $('#confirm-topic').on('click', function() {
            let selectedTopic;
            
            // Check if a predefined topic is selected
            const $selectedItem = $('.autoarticle-topic-item.selected');
            if ($selectedItem.length) {
                selectedTopic = $selectedItem.text();
            } else {
                // Use custom topic if no predefined topic is selected
                selectedTopic = $customTopic.val().trim();
            }
            
            if (!selectedTopic) {
                showMessage('Please select or enter a topic', 'error', 'generation');
                return;
            }
            
            $modal.hide();
            generateArticle(selectedTopic, style, imageStyle, $submitBtn, $form);
        });
        
        // Handle cancel button
        $('#cancel-topic').on('click', function() {
            $modal.hide();
        });
        
        // Handle close button
        $('.autoarticle-modal-close').on('click', function() {
            $modal.hide();
        });
    }

    function generateArticle(topic, style, imageStyle, $submitBtn, $form) {
        showLoadingOverlay('Generating article...');
        
        // Prepare request parameters
        const requestParams = {
            external: 1,
            custom: topic,
            style: style,
            image_style: imageStyle,
            image_size: $('#image_size').val(),
            language: $('#language').val(),
            topic_usage: $('input[name="topic_usage"]:checked').val() || 'exact_match',
            custom_writing_prompt: $('#custom_writing_prompt').val(),
            custom_image_prompt: $('#custom_image_prompt').val()
        };
        
        // Add credentials if they exist
        if (autoarticle.user_email && storedPassword) {
            requestParams.email = autoarticle.user_email;
            requestParams.password = storedPassword;
        }
        
        $.ajax({
            url: autoarticle.api_url,
            type: 'GET',
            data: requestParams,
            success: function(response) {
                if (response.success) {
                    const slugData = {
                        use_topic_as_slug: $('#use_topic_as_slug').is(':checked'),
                        slug_prefix: $('#slug_prefix').val(),
                        slug_suffix: $('#slug_suffix').val(),
                        topic: topic
                    };
                    $.ajax({
                        url: autoarticle.ajax_url,
                        type: 'POST',
                        data: {
                            action: 'autoarticle_generate_article',
                            nonce: autoarticle.nonce,
                            article_data: response,
                            slug_data: slugData
                        },
                        success: function(result) {
                            if (result.success) {
                                if (response.log_id) {
                                    $.post('/autoarticle-folder/update-log.php', {
                                        log_id: response.log_id,
                                        article_url: result.data.post_url,
                                        nonce: autoarticle.nonce
                                    });
                                }
                                showSuccessModal(result.data.post_url);
                                updateArticlesRemaining(result.data.articles_remaining);
                                // Refresh the count after a short delay to ensure it's up-to-date
                                setTimeout(getArticlesRemaining, 2000);
                            } else {
                                showMessage(result.data || 'Failed to create post', 'error', 'generation');
                            }
                            hideLoadingOverlay();
                        },
                        error: function() {
                            showMessage('Failed to create post', 'error', 'generation');
                            hideLoadingOverlay();
                            // Also refresh the count in case of error (article might have been generated regardless)
                            getArticlesRemaining();
                        }
                    });
                } else {
                    showMessage(response.error || 'Failed to generate article', 'error', 'generation');
                    hideLoadingOverlay();
                    // Refresh the count in case of error (quota might have changed)
                    getArticlesRemaining();
                }
            },
            error: function() {
                showMessage('Failed to connect to server', 'error', 'generation');
                hideLoadingOverlay();
                // Refresh the count in case of error
                getArticlesRemaining();
            }
        });
    }

    function showSuccessModal(postUrl) {
        const $modal = $('#autoarticle-success-modal');
        const $message = $('#success-message');
        const $viewLink = $('#view-article-link');
        
        $message.text('Article generated successfully!');
        $viewLink.attr('href', postUrl);
        $modal.show();
         
        // Set up click handlers for both close buttons
        $('.autoarticle-success-modal-close, #close-success-modal').on('click', function() {
            $modal.hide();
            // Refresh the articles count when closing the success modal
            getArticlesRemaining();
        });
    }

    // Handle style selection changes
    $('.style-select').on('change', function() {
        const $select = $(this);
        const value = $select.val();
        const description = $select.find('option:selected').data('description');
        const descriptionContainer = $select.siblings('.style-description');
        const styleType = $select.attr('id') === 'style' ? 'writing' :
                         $select.attr('id') === 'image_style' ? 'image' :
                         $select.attr('id') === 'language' ? 'language' :
                         $select.attr('id') === 'post_status' ? 'post_status' :
                         'image_size';
        
        // Update description
        if (descriptionContainer.length) {
            descriptionContainer.text(description).addClass('active');
        }

        // Show loading overlay with 'Saving...' message
        showLoadingOverlay('Saving...');

        // Save the selected style
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: styleType === 'writing' ? 'autoarticle_save_writing_style' :
                        styleType === 'image' ? 'autoarticle_save_image_style' :
                        styleType === 'language' ? 'autoarticle_save_language' :
                        styleType === 'post_status' ? 'autoarticle_save_post_status' :
                        'autoarticle_save_image_size',
                nonce: autoarticle.nonce,
                style: value,
                post_status: styleType === 'post_status' ? value : undefined
            },
            success: function() {
                hideLoadingOverlay();
            },
            error: function() {
                hideLoadingOverlay();
                showMessage('Failed to save style preference', 'error', 'generation');
            }
        });
    });

    // Set initial style descriptions
    $('#style').each(function() {
        const description = $(this).find('option:selected').data('description');
        if (description) {
            $(this).siblings('.writing-style-description').text(description).addClass('active');
        }
    });
    
    $('#image_style').each(function() {
        const description = $(this).find('option:selected').data('description');
        if (description) {
            $(this).siblings('.image-style-description').text(description).addClass('active');
        }
    });

    // Topics handling
    const topicsInput = $('#topic');
    const topicsPills = $('.topics-pills');
    const topicsHidden = $('#topics');
    let topics = [];
    let isInitializing = true; // Add flag to track initialization

    // Load saved topics on page load
    const savedTopics = topicsHidden.val() ? JSON.parse(topicsHidden.val()) : [];
    // Clear any existing pills
    topicsPills.empty();
    topics = [];
    // Add each unique topic
    savedTopics.forEach(topic => {
        if (topic && typeof topic === 'string') {
            const sanitizedTopic = sanitizeTopic(topic);
            if (sanitizedTopic && sanitizedTopic.length >= 3) {
                topics.push(sanitizedTopic);
                const pill = $(`
                    <div class="topic-pill">
                        <span>${sanitizedTopic}</span>
                        <span class="remove-topic dashicons dashicons-no-alt"></span>
                    </div>
                `);
                topicsPills.append(pill);
            }
        }
    });

    // Set initialization complete after a short delay to ensure DOM is ready
    setTimeout(() => {
        isInitializing = false;
    }, 1000);

    function updateTopics() {
        // Don't save during initialization
        if (isInitializing) {
            return;
        }

        // Ensure topics is an array
        if (!Array.isArray(topics)) {
            topics = [];
        }
        
        // Remove any invalid topics
        topics = topics.filter(topic => validateTopic(topic));
        
        // Update hidden input
        topicsHidden.val(JSON.stringify(topics));
        
        // Save topics to database
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_topics',
                nonce: autoarticle.nonce,
                topics: topics
            },
            success: function(response) {
                if (response.success && response.data && response.data.topics) {
                    // Update topics array with server response
                    topics = response.data.topics;
                    // Update hidden input
                    topicsHidden.val(JSON.stringify(topics));
                    // Update UI
                    topicsPills.empty();
                    topics.forEach(topic => {
                        const pill = $(`
                            <div class="topic-pill">
                                <span>${topic}</span>
                                <span class="remove-topic dashicons dashicons-no-alt"></span>
                            </div>
                        `);
                        topicsPills.append(pill);
                    });
                }
                hideLoadingOverlay();
                // Trigger frequency change to update cron
                $('#frequency').trigger('change');
            },
            error: function() {
                hideLoadingOverlay();
                showMessage('Failed to save topics', 'error', 'generation');
            }
        });
    }

    function sanitizeTopic(topic) {
        if (!topic || typeof topic !== 'string') {
            return '';
        }
        
        // Trim whitespace
        topic = topic.trim();
        
        // Remove any HTML tags
        topic = topic.replace(/<[^>]*>/g, '');
        
        // Remove any SQL injection attempts
        topic = topic.replace(/['";]/g, '');
        
        return topic;
    }

    function validateTopic(topic) {
        if (!topic || typeof topic !== 'string') {
            return false;
        }
        
        topic = sanitizeTopic(topic);
        
        if (topic.length < 3) {
            showMessage('Topics must be at least 3 characters long', 'error', 'generation');
            return false;
        }
        
        return topic;
    }

    function addTopic(topic) {
        topic = validateTopic(topic);
        if (!topic) {
            hideLoadingOverlay();
            return;
        }
        
        // Check if topic already exists in the pills
        if ($('.topic-pill span:first-child').filter(function() {
            return $(this).text() === topic;
        }).length > 0) {
            hideLoadingOverlay();
            return;
        }
        
        topics.push(topic);
        const pill = $(`
            <div class="topic-pill">
                <span>${topic}</span>
                <span class="remove-topic dashicons dashicons-no-alt"></span>
            </div>
        `);
        topicsPills.append(pill);
        updateTopics();
    }

    function removeTopic(topic) {
        topics = topics.filter(t => t !== topic);
        $('.topic-pill').filter(function() {
            return $(this).find('span:first-child').text() === topic;
        }).remove();
        updateTopics();
    }

    // Handle topic input
    topicsInput.off('keydown').on('keydown', function(e) {
        if (e.key === 'Enter' || e.key === ',') {
            e.preventDefault();
            const topic = $(this).val();
            if (topic) {
                showLoadingOverlay('Saving...');
                addTopic(topic);
                $(this).val('');
            }
        }
    });

    // Handle click on the save topic button
    $('#save-topic-button').on('click', function() {
        const topic = topicsInput.val();
        if (topic) {
            showLoadingOverlay('Saving...');
            addTopic(topic);
            topicsInput.val('');
        }
    });

    // Handle topic removal
    topicsPills.off('click').on('click', '.remove-topic', function() {
        const pill = $(this).closest('.topic-pill');
        const topic = pill.find('span:first').text();
        showLoadingOverlay('Saving...');
        removeTopic(topic);
        pill.remove();
    });

    // Handle frequency changes
    $('#frequency').on('change.adminJsFrequency', function() {
        const frequency = $(this).val();
        const time = $('#schedule_time').val() || '09:00'; // Default to 9:00 if no time set
        // Show loading overlay
        showLoadingOverlay();
        
        let ajaxData = {
            action: 'autoarticle_save_schedule',
            nonce: autoarticle.nonce,
            frequency: frequency,
            schedule_time: time
        };

        // Check if this is from the frontend shortcode
        if ($(this).closest('.autoarticle-frontend-wrapper').length > 0) {
            ajaxData.source = 'frontend_shortcode';
        }

        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: ajaxData,
            success: function(response) {
                hideLoadingOverlay();
                // Update next run time display
                if (response.data && response.data.next_run) {
                    const nextRun = new Date(response.data.next_run);
                    const formattedDate = nextRun.toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric',
                        hour: 'numeric',
                        minute: 'numeric',
                        hour12: true
                    });
                    $('.next-run-time').text(formattedDate);
                } else {
                    $('.next-run-time').text('Not scheduled');
                }
            },
            error: function() {
                hideLoadingOverlay();
                showMessage('Failed to save schedule', 'error', 'generation');
            }
        });
    });

    // Handle time changes
    $('#schedule_time').on('change', function() {
        const time = $(this).val() || '09:00'; // Default to 9:00 if no time set
        const frequency = $('#frequency').val();
        
        // Validate time format
        if (!time.match(/^([0-1]?[0-9]|2[0-3]):[0-5][0-9]$/)) {
            showMessage('Invalid time format. Please use HH:MM format.', 'error', 'generation');
            return;
        }
        
        // Show loading overlay
        showLoadingOverlay();
        
        let ajaxData = {
            action: 'autoarticle_save_schedule',
            nonce: autoarticle.nonce,
            frequency: frequency,
            schedule_time: time
        };

        // Check if this is from the frontend shortcode
        if ($(this).closest('.autoarticle-frontend-wrapper').length > 0) {
            ajaxData.source = 'frontend_shortcode';
        }

        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: ajaxData,
            success: function(response) {
                hideLoadingOverlay();
                // Update next run time display
                if (response.data && response.data.next_run) {
                    const nextRun = new Date(response.data.next_run);
                    const formattedDate = nextRun.toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric',
                        hour: 'numeric',
                        minute: 'numeric',
                        hour12: true
                    });
                    $('.next-run-time').text(formattedDate);
                } else {
                    $('.next-run-time').text('Not scheduled');
                }
            },
            error: function() {
                hideLoadingOverlay();
                showMessage('Failed to save schedule', 'error', 'generation');
            }
        });
    });

    function generateNow() {
        // Show the modal regardless of topics availability
        $('#autoarticle-topic-modal').show();
        
        // If there are topics, populate them
        if (topics.length > 0) {
            let topicsHtml = '';
            topics.forEach((topic, index) => {
                topicsHtml += `<div class="topic-item">
                    <input type="radio" name="selected-topic" id="topic-${index}" value="${topic}">
                    <label for="topic-${index}">${topic}</label>
                </div>`;
            });
            $('#topics-container').html(topicsHtml);
        } else {
            $('#topics-container').html('<p>No trending topics available. Please use the custom term box below.</p>');
        }
    }

    // Handle topic usage changes
    $('input[name="topic_usage"]').on('change', function() {
        const value = $(this).val();
        
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_topic_usage',
                nonce: autoarticle.nonce,
                topic_usage: value
            },
            success: function(response) {
                if (!response.success) {
                    showMessage('Failed to save topic usage preference', 'error', 'generation');
                }
            },
            error: function() {
                showMessage('Failed to save topic usage preference', 'error', 'generation');
            }
        });
    });

    // Handle writing style changes
    $('#style').on('change', function() {
        const value = $(this).val();
        const description = $(this).find('option:selected').data('description');
        $(this).siblings('.writing-style-description').text(description).addClass('active');
        
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_writing_style',
                nonce: autoarticle.nonce,
                style: value
            }
        });
    });

    // Handle image style changes
    $('#image_style').on('change', function() {
        const value = $(this).val();
        const description = $(this).find('option:selected').data('description');
        $(this).siblings('.image-style-description').text(description).addClass('active');
        
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_image_style',
                nonce: autoarticle.nonce,
                style: value
            }
        });
    });

    // Handle language changes
    $('#language').on('change', function() {
        const value = $(this).val();
        
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_language',
                nonce: autoarticle.nonce,
                style: value
            }
        });
    });

    // Handle image size changes
    $('#image_size').on('change', function() {
        const value = $(this).val();
        
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_image_size',
                nonce: autoarticle.nonce,
                style: value
            }
        });
    });

    // Handle trigger overdue button click
    $('#autoarticle-trigger-overdue').on('click', function() {
        const $button = $(this);
        $button.prop('disabled', true).text('Triggering WP Cron...');
        showMessage('Attempting to trigger WP Cron runner...', 'info', 'generation');
        
        // Make a simple GET request to wp-cron.php to trigger the cron runner
        $.ajax({
            url: '/wp-cron.php?doing_wp_cron', // Standard WP cron trigger URL
            type: 'GET',
            timeout: 60000, // Add a timeout (10 seconds)
            cache: false, // Prevent caching
            success: function() {
                 showMessage('WP Cron process triggered successfully. Overdue jobs should run shortly if scheduled correctly.', 'success', 'generation');
            },
            error: function(jqXHR, textStatus, errorThrown) {
                 // Note: This request might often "fail" or time out from the browser's perspective 
                 // because wp-cron.php doesn't necessarily return a standard success response, 
                 // especially if it redirects or takes time. Success here just means the request was sent.
                 // We interpret any response (even error/timeout) as the trigger likely being sent.
                 showMessage('WP Cron trigger request sent. Check for new articles shortly.', 'info', 'generation');
                 console.warn('WP Cron trigger request status:', textStatus, errorThrown);
            },
            complete: function() {
                // Re-enable button relatively quickly, as the actual cron jobs run in the background
                $button.prop('disabled', false).text('Trigger Overdue Articles');
            }
        });
    });

    // Handle accordion functionality
    $('.autoarticle-accordion-header').on('click', function() {
        const $header = $(this);
        const $content = $header.next('.autoarticle-accordion-content');
        
        $header.toggleClass('active');
        $content.slideToggle(200);
    });

    // Handle advanced prompts toggle
    $('.advanced-prompts-toggle').on('click', function(e) {
        e.preventDefault();
        $(this).next('.advanced-prompts-content').slideToggle(200);
    });

    // Handle saving custom prompts
    $('#custom_writing_prompt, #custom_image_prompt').on('change', function() {
        const $textarea = $(this);
        const value = $textarea.val();
        const type = $textarea.attr('id');

        showLoadingOverlay('Saving...');
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_custom_prompt',
                nonce: autoarticle.nonce,
                prompt_type: type,
                prompt_value: value
            },
            success: function() {
                hideLoadingOverlay();
            },
            error: function() {
                hideLoadingOverlay();
                showMessage('Failed to save custom prompt', 'error', 'generation');
            }
        });
    });

    // Handle saving slug settings
    $('#use_topic_as_slug, #slug_prefix, #slug_suffix').on('change', function() {
        showLoadingOverlay('Saving...');
        $.ajax({
            url: autoarticle.ajax_url,
            type: 'POST',
            data: {
                action: 'autoarticle_save_slug_settings',
                nonce: autoarticle.nonce,
                use_topic_as_slug: $('#use_topic_as_slug').is(':checked') ? 1 : 0,
                slug_prefix: $('#slug_prefix').val(),
                slug_suffix: $('#slug_suffix').val()
            },
            success: function() {
                hideLoadingOverlay();
            },
            error: function() {
                hideLoadingOverlay();
                showMessage('Failed to save slug settings', 'error', 'generation');
            }
        });
    });

    // Sanitize slug prefix and suffix fields
    $('#slug_prefix, #slug_suffix').on('input', function() {
        const sanitizedValue = $(this).val().replace(/[^a-zA-Z0-9_-]/g, '');
        $(this).val(sanitizedValue);
    });
}); 