<?php
class AutoArticle_Admin {
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    public function add_admin_menu() {
        add_menu_page(
            'AutoArticle Generator',
            'AutoArticle',
            'manage_options',
            'ai-automatic-blog-posts',
            array($this, 'render_admin_page'),
            'dashicons-welcome-write-blog',
            6
        );
    }

    public function enqueue_admin_assets($hook) {
        if ($hook !== 'toplevel_page_ai-automatic-blog-posts') return;

        wp_enqueue_style(
            'autoarticle-admin',
            AUTOARTICLE_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            AUTOARTICLE_VERSION
        );

        wp_enqueue_script(
            'autoarticle-admin',
            AUTOARTICLE_PLUGIN_URL . 'admin/js/admin.js',
            array('jquery'),
            AUTOARTICLE_VERSION,
            true
        );

        $user_id = get_current_user_id();
        $email = get_user_meta($user_id, 'autoarticle_membership_email', true);
        $password = get_user_meta($user_id, 'autoarticle_password', true);
        $is_logged_in = !empty($email) && !empty($password);
        
        wp_localize_script('autoarticle-admin', 'autoarticle', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('autoarticle_nonce'),
            'prose_url' => 'https://autoarticle.net',
            'api_url' => 'https://autoarticle.net/autoarticle-folder/autoarticle-new.php',
            'user_email' => $email,
            'user_password' => $password,
            'is_logged_in' => $is_logged_in
        ));
    }

    public function render_admin_page($is_shortcode_context = false) {
        $user_id = get_current_user_id();
        $membership_email = get_user_meta($user_id, 'autoarticle_membership_email', true);
        $saved_topics = get_user_meta($user_id, 'autoarticle_topics', true);
        if (is_string($saved_topics)) {
            $saved_topics = json_decode($saved_topics, true);
        }
        if (!is_array($saved_topics)) {
            $saved_topics = array();
        }
        $topics_json = json_encode($saved_topics);
        $saved_writing_style = get_user_meta($user_id, 'autoarticle_writing_style', true);
        $saved_image_style = get_user_meta($user_id, 'autoarticle_image_style', true);
        $saved_language = get_user_meta($user_id, 'autoarticle_language', true) ?: 'en';
        $saved_image_size = get_user_meta($user_id, 'autoarticle_image_size', true) ?: 'square';
        $saved_topic_usage = get_user_meta($user_id, 'autoarticle_topic_usage', true) ?: 'exact_match';
        $saved_post_status = get_user_meta($user_id, 'autoarticle_post_status', true) ?: 'publish';
        $saved_custom_writing_prompt = get_user_meta($user_id, 'custom_writing_prompt', true);
        $saved_custom_image_prompt = get_user_meta($user_id, 'custom_image_prompt', true);
        
        // Slug settings
        $saved_use_topic_as_slug = get_user_meta($user_id, 'autoarticle_use_topic_as_slug', true);
        $saved_slug_prefix = get_user_meta($user_id, 'autoarticle_slug_prefix', true);
        $saved_slug_suffix = get_user_meta($user_id, 'autoarticle_slug_suffix', true);
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'autoarticle_scheduled';
        $scheduled = $wpdb->get_row($wpdb->prepare("SELECT frequency, next_run, topic_usage FROM $table_name WHERE user_id = %d", $user_id));
        $saved_frequency = $scheduled ? $scheduled->frequency : '';
        $saved_time = $scheduled && $scheduled->next_run ? gmdate('H:i', strtotime($scheduled->next_run)) : '09:00';
        $saved_topic_usage = $scheduled ? $scheduled->topic_usage : $saved_topic_usage;
        $is_logged_in = !empty($membership_email);
        ?>
        <div class="wrap">
            <h1>AutoArticle Generator</h1>
            <div class="autoarticle-generator-form">
                <div class="autoarticle-header">
                    <div class="autoarticle-header-left">
                        <div class="autoarticle-membership-info">
                            <div class="membership-level-row">
                                <span class="membership-level">Membership Level: <span id="membership-level-name">Loading...</span></span>
                                <button type="button" id="refresh-membership-status" class="button button-small" style="display: none;">Refresh</button>
                            </div>
                            <span class="articles-remaining">Articles Remaining This Month: <span id="articles-remaining">Loading...</span></span>
                            <div id="upgrade-button-container"><?php /*<!-- Upgrade button will be added here by JavaScript if needed -->*/ ?></div>
                        </div>
                    </div>
                    <div class="autoarticle-header-right">
                        <?php if ($is_logged_in && is_admin()): ?>
                            <button type="button" id="autoarticle-logout" class="button">Logout</button>
                        <?php elseif (is_admin()): ?>
                            <button type="button" id="autoarticle-login-btn" class="button button-primary">Login</button>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="autoarticle-how-to-use">
                    <h3 class="autoarticle-accordion-header">How to use AutoArticle <span class="dashicons dashicons-arrow-down-alt2"></span></h3>
                    <div class="autoarticle-accordion-content" style="display: none;">
                        <div class="how-to-steps">
                            <div class="step"><span class="step-number">1</span><span class="step-text">Enter your topics</span></div>
                            <div class="step"><span class="step-number">2</span><span class="step-text">Select your writing & image style</span></div>
                            <div class="step"><span class="step-number">3</span><span class="step-text">Set a schedule, or click "Generate Now"</span></div>
                        </div>
                    </div>
                </div>

                <form id="autoarticle-generate">
                    <input type="hidden" id="autoarticle-password" name="password" value="<?php echo esc_attr(get_user_meta($user_id, 'autoarticle_password', true)); ?>">
                    <p>
                        <label for="topic">
                            <span class="step-number-blue">1</span> Topics<br>
                            <small>Topics will be cycled through if using the schedule, or available to select when manually generating.</small>
                        </label>
                        <div class="topics-container">
                            <div class="topics-input-wrapper">
                                <input type="text" id="topic" name="topic" placeholder="Type a topic and press Enter">
                                <button type="button" id="save-topic-button" class="button">Save</button>
                                <div class="topics-pills"></div>
                            </div>
                            <input type="hidden" id="topics" name="topics" value="<?php echo esc_attr($topics_json); ?>">
                        </div>
                        <div class="topic-usage-toggle" style="display: none;">
                            <label>Topic Usage</label>
                            <div class="toggle-switch">
                                <input type="radio" id="topic_usage_relevant" name="topic_usage" value="relevant_articles" <?php checked($saved_topic_usage, 'relevant_articles'); ?>>
                                <label for="topic_usage_relevant" class="tooltip-trigger" data-tooltip="Create varied but relevant articles">More Variety</label>
                                <input type="radio" id="topic_usage_exact" name="topic_usage" value="exact_match" <?php checked($saved_topic_usage, 'exact_match'); ?>>
                                <label for="topic_usage_exact" class="tooltip-trigger" data-tooltip="Ensure articles always contains topic term">Less Variety</label>
                            </div>
                        </div>
                    </p>

                    <div class="style-selectors">
                        <div class="style-selector">
                            <label for="style"><span class="step-number-blue">2</span> Writing Style</label>
                            <select id="style" name="style" class="style-select">
                                <option value="professional" <?php selected($saved_writing_style, 'professional'); ?> data-description="Write in a formal, professional tone suitable for business or academic audiences. Use precise language and maintain a serious, authoritative voice.">Professional</option>
                                <option value="conversational" <?php selected($saved_writing_style, 'conversational'); ?> data-description="Write in a friendly, conversational tone as if speaking directly to the reader. Use casual language and include relatable examples.">Conversational</option>
                                <option value="journalistic" <?php selected($saved_writing_style, 'journalistic'); ?> data-description="Write in a clear, objective news style. Focus on facts, use the inverted pyramid structure, and maintain neutrality.">Journalistic</option>
                                <option value="storytelling" <?php selected($saved_writing_style, 'storytelling'); ?> data-description="Write in a narrative style that tells a story. Use descriptive language, build tension, and create an engaging narrative arc.">Storytelling</option>
                                <option value="analytical" <?php selected($saved_writing_style, 'analytical'); ?> data-description="Write in a detailed, analytical style. Focus on data, patterns, and in-depth analysis of the topic.">Analytical</option>
                                <option value="educational" <?php selected($saved_writing_style, 'educational'); ?> data-description="Write in an instructive, explanatory style. Break down complex concepts and use examples to aid understanding.">Educational</option>
                                <option value="opinion" <?php selected($saved_writing_style, 'opinion'); ?> data-description="Write in a persuasive, opinionated style. Present a clear viewpoint while acknowledging other perspectives.">Opinion</option>
                                <option value="technical" <?php selected($saved_writing_style, 'technical'); ?> data-description="Write in a precise, technical style. Use industry-specific terminology and focus on technical details.">Technical</option>
                                <option value="humorous" <?php selected($saved_writing_style, 'humorous'); ?> data-description="Write in a light-hearted, witty style. Include appropriate humor while maintaining the article's informative value.">Humorous</option>
                                <option value="inspirational" <?php selected($saved_writing_style, 'inspirational'); ?> data-description="Write in an uplifting, motivational style. Focus on positive aspects and include encouraging messages.">Inspirational</option>
                                <option value="random" <?php selected($saved_writing_style, 'random'); ?> data-description="Let the AI choose a random writing style for variety.">Random</option>
                            </select>
                            <div class="style-description writing-style-description"></div>
                        </div>

                        <div class="style-selector">
                            <label for="language">Language</label>
                            <select id="language" name="language" class="style-select">
                                <option value="us" <?php selected($saved_language, 'us'); ?>>🇺🇸 US English</option>
                                <option value="en" <?php selected($saved_language, 'en'); ?>>🇬🇧 English</option>
                                <option value="es" <?php selected($saved_language, 'es'); ?>>🇪🇸 Spanish</option>
                                <option value="fr" <?php selected($saved_language, 'fr'); ?>>🇫🇷 French</option>
                                <option value="de" <?php selected($saved_language, 'de'); ?>>🇩🇪 German</option>
                                <option value="it" <?php selected($saved_language, 'it'); ?>>🇮🇹 Italian</option>
                                <option value="pt" <?php selected($saved_language, 'pt'); ?>>🇵🇹 Portuguese</option>
                                <option value="nl" <?php selected($saved_language, 'nl'); ?>>🇳🇱 Dutch</option>
                                <option value="pl" <?php selected($saved_language, 'pl'); ?>>🇵🇱 Polish</option>
                                <option value="ru" <?php selected($saved_language, 'ru'); ?>>🇷🇺 Russian</option>
                                <option value="ja" <?php selected($saved_language, 'ja'); ?>>🇯🇵 Japanese</option>
                                <option value="zh" <?php selected($saved_language, 'zh'); ?>>🇨🇳 Chinese</option>
                                <option value="ar" <?php selected($saved_language, 'ar'); ?>>🇸🇦 Arabic</option>
                                <option value="hi" <?php selected($saved_language, 'hi'); ?>>🇮🇳 Hindi</option>
                                <option value="ko" <?php selected($saved_language, 'ko'); ?>>🇰🇷 Korean</option>
                                <option value="tr" <?php selected($saved_language, 'tr'); ?>>🇹🇷 Turkish</option>
                            </select>
                        </div>

                        <div class="style-selector">
                            <label for="image_style">Image Style</label>
                            <select id="image_style" name="image_style" class="style-select">
                                <option value="photograph" <?php selected($saved_image_style, 'photograph'); ?> data-description="Create a photorealistic image that looks like it was taken with a camera.">Photograph</option>
                                <option value="realistic" <?php selected($saved_image_style, 'realistic'); ?> data-description="Create a highly detailed, photorealistic image with accurate lighting and textures.">Realistic</option>
                                <option value="artistic" <?php selected($saved_image_style, 'artistic'); ?> data-description="Create an artistic interpretation with creative composition and stylized elements.">Artistic</option>
                                <option value="minimalist" <?php selected($saved_image_style, 'minimalist'); ?> data-description="Create a clean, minimalist image with simple shapes and limited color palette.">Minimalist</option>
                                <option value="vintage" <?php selected($saved_image_style, 'vintage'); ?> data-description="Create an image with a vintage aesthetic, using muted colors and classic composition.">Vintage</option>
                                <option value="cinematic" <?php selected($saved_image_style, 'cinematic'); ?> data-description="Create a cinematic image with dramatic lighting and wide-angle composition.">Cinematic</option>
                                <option value="sketch" <?php selected($saved_image_style, 'sketch'); ?> data-description="Create an image that looks like a detailed pencil or charcoal sketch.">Sketch</option>
                                <option value="watercolor" <?php selected($saved_image_style, 'watercolor'); ?> data-description="Create an image with the soft, flowing aesthetic of watercolor painting.">Watercolor</option>
                                <option value="digital_art" <?php selected($saved_image_style, 'digital_art'); ?> data-description="Create a modern digital art style image with vibrant colors and clean lines.">Digital Art</option>
                                <option value="oil_painting" <?php selected($saved_image_style, 'oil_painting'); ?> data-description="Create an image that mimics the rich textures and brushstrokes of oil painting.">Oil Painting</option>
                                <option value="cartoon" <?php selected($saved_image_style, 'cartoon'); ?> data-description="Create a stylized cartoon image with bold lines and vibrant colors.">Cartoon</option>
                                <option value="random" <?php selected($saved_image_style, 'random'); ?> data-description="Let the AI choose a random image style for variety.">Random</option>
                            </select>
                            <div class="style-description image-style-description"></div>
                        </div>

                        <div class="style-selector">
                            <label for="image_size">Image Size</label>
                            <select id="image_size" name="image_size" class="style-select">
                                <option value="square" <?php selected($saved_image_size, 'square'); ?>>Square (1024x1024)</option>
                                <option value="landscape" <?php selected($saved_image_size, 'landscape'); ?>>Landscape (1792x1024)</option>
                                <option value="portrait" <?php selected($saved_image_size, 'portrait'); ?>>Portrait (1024x1792)</option>
                            </select>
                        </div>

                        <div class="style-selector">
                            <label for="post_status">Article Status</label>
                            <select id="post_status" name="post_status" class="style-select">
                                <option value="publish" <?php selected($saved_post_status, 'publish'); ?>>Published</option>
                                <option value="draft" <?php selected($saved_post_status, 'draft'); ?>>Draft</option>
                                <?php if (!$is_shortcode_context): ?>
                                    <option value="pending" <?php selected($saved_post_status, 'pending'); ?>>Pending Review</option>
                                    <option value="private" <?php selected($saved_post_status, 'private'); ?>>Private</option>
                                <?php endif; ?>
                            </select>
                        </div>
                    </div>

                    <div class="advanced-prompts-wrapper">
                        <a href="#" class="advanced-prompts-toggle">Advanced</a>
                        <div class="advanced-prompts-content" style="display: none;">
                            <div class="prompt-field">
                                <label for="custom_writing_prompt">Custom Writing Prompt</label>
                                <textarea id="custom_writing_prompt" name="custom_writing_prompt" maxlength="300" placeholder="e.g., 'Focus on the benefits for small businesses...'"><?php echo esc_textarea($saved_custom_writing_prompt); ?></textarea>
                            </div>
                            <div class="prompt-field">
                                <label for="custom_image_prompt">Custom Image Prompt</label>
                                <textarea id="custom_image_prompt" name="custom_image_prompt" maxlength="300" placeholder="e.g., 'A close-up of a person smiling...'"><?php echo esc_textarea($saved_custom_image_prompt); ?></textarea>
                            </div>
                             <div class="prompt-field">
                                <h4>Custom URL Slug</h4>
                                <label>
                                    <input type="checkbox" id="use_topic_as_slug" name="use_topic_as_slug" value="1" <?php checked($saved_use_topic_as_slug, 1); ?>>
                                    Use Topic as URL Slug
                                </label>
                            </div>
                            <div class="prompt-field">
                                <label for="slug_prefix">Prefix</label>
                                <input type="text" id="slug_prefix" name="slug_prefix" value="<?php echo esc_attr($saved_slug_prefix); ?>" placeholder="e.g., 'news-'">
                            </div>
                            <div class="prompt-field">
                                <label for="slug_suffix">Suffix</label>
                                <input type="text" id="slug_suffix" name="slug_suffix" value="<?php echo esc_attr($saved_slug_suffix); ?>" placeholder="e.g., '-update'">
                            </div>
                        </div>
                    </div>

                    <div class="autoarticle-generate-section">
                        <h2><span class="step-number-blue">3</span> Generate Articles</h2>
                        <div class="autoarticle-generate-sections">
                            <div class="autoarticle-schedule-section">
                                <h3>Set a Schedule</h3>
                                <p class="autoarticle-section-description">Configure automatic article generation based on your preferred frequency.</p>
                                <div class="autoarticle-schedule-content">
                                    <label for="frequency">Frequency:<br><small>(Scheduling available on Premium and Pro plans)</small></label>
                                    <select id="frequency" name="frequency" disabled>
                                        <option value="" <?php selected($saved_frequency, ''); ?>>Disabled</option>
                                        <option value="twicedaily" <?php selected($saved_frequency, 'twicedaily'); ?>>Twice Daily</option>
                                        <option value="daily" <?php selected($saved_frequency, 'daily'); ?>>Daily</option>
                                        <option value="weekly" <?php selected($saved_frequency, 'weekly'); ?>>Weekly</option>
                                        <option value="monthly" <?php selected($saved_frequency, 'monthly'); ?>>Monthly</option>
                                    </select>
                                    <input type="time" id="schedule_time" name="schedule_time" value="<?php echo esc_attr($saved_time); ?>" disabled>
                                    <?php if ($saved_frequency): ?>
                                        <div class="schedule-status active">
                                            <p>Next Article will be generated on: <span class="next-run-time"><?php echo $scheduled ? esc_html(gmdate('F j, Y, g:i a', strtotime($scheduled->next_run))) : 'Not scheduled'; ?></span></p>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="autoarticle-manual-section">
                                <h3>OR Manually Generate</h3>
                                <button type="button" id="autoarticle-generate-now" class="button button-secondary">Generate Now</button>
                            </div>
                        </div>
                    </div>
                </form>

                <div id="generation-message" class="notice" style="display: none;"></div>
                <div class="autoarticle-loading-overlay" style="display: none;">
                    <div class="autoarticle-spinner"></div>
                    <div class="autoarticle-loading-text">Generating article...</div>
                </div>
            </div>
        </div>

        <div id="autoarticle-login-modal" class="autoarticle-modal" style="display: none;">
            <div class="autoarticle-modal-content">
                <span class="autoarticle-modal-close">&times;</span>
                <h2>Login to Your AutoArticle.net Account</h2>
                <p>Upgrade to a paid plan to unlock additional features and increased article generation limits.</p>
                <form id="autoarticle-login">
                    <p><label for="email">Email:</label><input type="email" id="email" name="email" required></p>
                    <p><label for="password">Password:</label><input type="password" id="password" name="password" required></p>
                    <div id="login-message" class="notice" style="display: none;"></div>
                    <div class="autoarticle-modal-actions">
                        <button type="submit" class="button button-primary">Login</button>
                        <a href="https://autoarticle.net/membership-account/membership-levels/" class="button" target="_blank">Create Account</a>
                        <button type="button" class="button" id="cancel-login">Cancel</button>
                    </div>
                </form>
            </div>
        </div>

        <div id="autoarticle-topic-modal" class="autoarticle-modal" style="display: none;">
            <div class="autoarticle-modal-content">
                <span class="autoarticle-modal-close">&times;</span>
                <h2>Select a Topic</h2>
                <div class="autoarticle-topic-list"></div>
                <div class="autoarticle-custom-topic">
                    <h3>Or Enter a Custom Topic</h3>
                    <input type="text" id="custom-topic" placeholder="Enter a custom topic">
                </div>
                <div class="autoarticle-modal-actions">
                    <button type="button" class="button button-primary" id="confirm-topic">Generate Article</button>
                    <button type="button" class="button" id="cancel-topic">Cancel</button>
                </div>
            </div>
        </div>

        <div id="autoarticle-success-modal" class="autoarticle-success-modal" style="display: none;">
            <div class="autoarticle-success-modal-content">
                <span class="autoarticle-success-modal-close">&times;</span>
                <h2>Success!</h2>
                <p id="success-message"></p>
                <div class="autoarticle-success-modal-actions">
                    <a href="" id="view-article-link" class="button button-primary" target="_blank">View Article</a>
                    <button type="button" class="button" id="close-success-modal">Close</button>
                </div>
            </div>
        </div>
        <?php
    }

    private function get_membership_name($level) {
        $levels = array(
            1 => 'Free',
            2 => 'Premium',
            3 => 'Pro'
        );
        return isset($levels[$level]) ? $levels[$level] : 'Unknown';
    }

    private function get_remaining_articles($user_id) {
        $email = get_user_meta($user_id, 'autoarticle_membership_email', true);
        $password = get_user_meta($user_id, 'autoarticle_password', true);
        
        if (!$email || !$password) {
            return 0;
        }
        
        $api = new AutoArticle_API();
        return $api->get_articles_remaining($email, $password);
    }
} 