<?php
/**
 * Plugin Name: AI Automatic Blog Posts
 * Plugin URI: https://autoarticle.net
 * Description: Generate AI-powered articles with customizable styles and scheduling
 * Version: 1.0.6
 * Author: AutoArticle.net
 * Text Domain: ai-automatic-blog-posts
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.0
 * Requires PHP: 7.2
 */

if (!defined('ABSPATH')) exit;

// Define plugin constants
define('AUTOARTICLE_VERSION', '1.0.0');
define('AUTOARTICLE_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AUTOARTICLE_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AUTOARTICLE_API_URL', 'https://autoarticle.net/autoarticle-folder/autoarticle-new.php');
define('AUTOARTICLE_DB_VERSION', '1.1'); // DB version for schema updates

// Include required files
require_once AUTOARTICLE_PLUGIN_DIR . 'includes/class-autoarticle-generator.php';
require_once AUTOARTICLE_PLUGIN_DIR . 'includes/class-autoarticle-admin.php';
require_once AUTOARTICLE_PLUGIN_DIR . 'includes/class-autoarticle-cron.php';
require_once AUTOARTICLE_PLUGIN_DIR . 'includes/class-autoarticle-api.php';

// Add custom cron schedules for weekly and monthly intervals
function autoarticle_add_cron_schedules($schedules) {
    if (!isset($schedules['weekly'])) {
        $schedules['weekly'] = array(
            'interval' => 604800, // 7 days in seconds
            'display' => __('Once Weekly', 'ai-automatic-blog-posts')
        );
    }
    if (!isset($schedules['monthly'])) {
        $schedules['monthly'] = array(
            'interval' => 2635200, // 30.5 days in seconds (average month)
            'display' => __('Once Monthly', 'ai-automatic-blog-posts')
        );
    }
    return $schedules;
}
add_filter('cron_schedules', 'autoarticle_add_cron_schedules');

// Initialize the plugin
function autoarticle_generator_init() {
    // Check for DB updates
    autoarticle_update_db_check();

    $plugin = new AutoArticle_Generator();
    $plugin->init();

    // Front-end functions
    // new AutoArticle_Frontend(); // This line will be moved
}
add_action('plugins_loaded', 'autoarticle_generator_init');

// Ensure the scheduled jobs cron is always running
// This runs hourly and checks the database for any jobs where next_run <= now
function autoarticle_ensure_cron_scheduled() {
    $hook = 'autoarticle_process_scheduled_jobs_hook';
    
    // Only schedule if not already scheduled
    if (!wp_next_scheduled($hook)) {
        wp_schedule_event(time(), 'hourly', $hook);
    }
}
add_action('init', 'autoarticle_ensure_cron_scheduled');

 
// Activation hook
register_activation_hook(__FILE__, 'autoarticle_generator_activate');
function autoarticle_generator_activate() {
    // Create necessary database tables
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    // Create scheduled jobs table
    $table_name = $wpdb->prefix . 'autoarticle_scheduled';
    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT, 
        user_id bigint(20) NOT NULL, 
        topics text NOT NULL, 
        style varchar(50) NOT NULL, 
        image_style varchar(50) NOT NULL, 
        language varchar(10) NOT NULL, 
        image_size varchar(20) NOT NULL, 
        frequency varchar(20) NOT NULL, 
        next_run datetime NOT NULL, 
        topic_usage varchar(20) NOT NULL DEFAULT 'exact_match', 
        post_status varchar(20) NOT NULL DEFAULT 'publish', 
        last_topic_index INT NOT NULL DEFAULT 0, 
        send_to_hs INT DEFAULT NULL, 
        use_topic_as_slug TINYINT(1) NOT NULL DEFAULT 0,
        slug_prefix TEXT DEFAULT NULL,
        slug_suffix TEXT DEFAULT NULL,
        custom_writing_prompt TEXT DEFAULT NULL,
        custom_image_prompt TEXT DEFAULT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    // Set the initial DB version
    add_option('autoarticle_db_version', AUTOARTICLE_DB_VERSION);

    // Schedule the cron job to run hourly for processing scheduled articles
    $hook = 'autoarticle_process_scheduled_jobs_hook';
    if (!wp_next_scheduled($hook)) {
        wp_schedule_event(time(), 'hourly', $hook);
    }

    add_option('autoarticle_do_activation_redirect', true);
    $current_user = wp_get_current_user();
    $username = 'N/A';
    $email = 'N/A';
    if ( $current_user instanceof WP_User && $current_user->ID != 0 ) {
        $username = $current_user->user_login;
        $email = $current_user->user_email;
    }
    $website_url = get_site_url();
    $ip_address = autoarticle_get_user_ip();
    $data_to_send = array(
        'action' => 'log_activation',
        'username' => $username,
        'email' => $email,
        'website' => $website_url,
        'ip_address_plugin' => $ip_address 
    );
    $remote_url = 'https://autoarticle.net/autoarticle-folder/autoarticle-new.php';
    $response = wp_remote_post( $remote_url, array(
        'method'    => 'POST',
        'timeout'   => 45,
        'redirection' => 5,
        'httpversion' => '1.0',
        'blocking'  => false, 
        'headers'   => array(),
        'body'      => $data_to_send,
        'cookies'   => array()
        )
    ); 
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'autoarticle_generator_deactivate');
function autoarticle_generator_deactivate() {
    // Clear any scheduled events
    wp_clear_scheduled_hook('autoarticle_process_scheduled_jobs_hook');
}

// Uninstall hook
register_uninstall_hook(__FILE__, 'autoarticle_generator_uninstall');
function autoarticle_generator_uninstall() {
    // Remove plugin data
    global $wpdb;
    $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}autoarticle_scheduled");
    
    // Remove user meta
    $wpdb->query("DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE 'autoarticle_%'");
    
    // Remove options
    delete_option('autoarticle_do_activation_redirect');
}

// Handle activation redirect
function autoarticle_generator_redirect() {
    if (get_option('autoarticle_do_activation_redirect', false)) {
        delete_option('autoarticle_do_activation_redirect');
        if (!isset($_GET['activate-multi'])) {
            wp_redirect(admin_url('admin.php?page=ai-automatic-blog-posts'));
            exit;
        }
    }
}
add_action('admin_init', 'autoarticle_generator_redirect');
 
// Check for database updates
function autoarticle_update_db_check() {
    $installed_ver = get_option('autoarticle_db_version');
    if ($installed_ver != AUTOARTICLE_DB_VERSION) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'autoarticle_scheduled';

        // Check and add slug columns
        $columns = $wpdb->get_col("DESC $table_name", 0);
        
        if (!in_array('use_topic_as_slug', $columns)) {
            $wpdb->query("ALTER TABLE $table_name ADD COLUMN use_topic_as_slug TINYINT(1) NOT NULL DEFAULT 0;");
        }
        if (!in_array('slug_prefix', $columns)) {
            $wpdb->query("ALTER TABLE $table_name ADD COLUMN slug_prefix TEXT DEFAULT NULL;");
        }
        if (!in_array('slug_suffix', $columns)) {
            $wpdb->query("ALTER TABLE $table_name ADD COLUMN slug_suffix TEXT DEFAULT NULL;");
        }

        if (!in_array('custom_writing_prompt', $columns)) {
            $wpdb->query("ALTER TABLE $table_name ADD COLUMN custom_writing_prompt TEXT DEFAULT NULL;");
        }
        if (!in_array('custom_image_prompt', $columns)) {
            $wpdb->query("ALTER TABLE $table_name ADD COLUMN custom_image_prompt TEXT DEFAULT NULL;");
        }

        update_option('autoarticle_db_version', AUTOARTICLE_DB_VERSION);
    }
}

function autoarticle_get_user_ip() {
    $ip = '';
    if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) { 
        $ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
    } elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) { 
        $ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
    } elseif ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
        $ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
    }
    return apply_filters( 'autoarticle_get_user_ip', $ip );
}
 

function autoarticle_settings_page_html() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
        <p>
            <?php esc_html_e( 'Welcome to AutoArticle. Settings will be here.', 'ai-automatic-blog-posts' ); ?>
        </p>
    </div>
    <?php
}
 
?> 