<?php
class AutoArticle_Cron {
    public function __construct() {
        // Add cron hook for the unified processing function
        add_action('autoarticle_process_scheduled_jobs_hook', array($this, 'process_scheduled_jobs')); 
        
        // Ensure HubSpot helper functions are available for cron jobs
        $hubspot_helpers_path = ABSPATH . 'autoarticle-folder/hubspot-helpers.php';
        if (file_exists($hubspot_helpers_path) && !function_exists('autoarticle_get_tokens_by_hubid')) {
            require_once $hubspot_helpers_path;
        }
    }

    public function process_scheduled_jobs() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'autoarticle_scheduled';
        $now = current_time('mysql');
        $query = $wpdb->prepare("SELECT * FROM $table_name WHERE next_run <= %s", $now);
        $jobs = $wpdb->get_results($query);
        if (empty($jobs)) return;
        foreach ($jobs as $job) {
            $this->process_job($job);
        }
    }

    public function process_job($job) {
        // Attempt to increase max execution time for this potentially long-running job
        if (function_exists('set_time_limit')) {
            @set_time_limit(600); // Set to 10 minutes for GPT-5 reasoning
        }
         
        error_log("[AutoArticle Cron] Processing job ID: {$job->id} for user ID: {$job->user_id}");

        global $wpdb;
        $table_name = $wpdb->prefix . 'autoarticle_scheduled';

        // Get topics and current index
        $topics = json_decode($job->topics, true);
        $current_index = $job->last_topic_index;

        if (empty($topics) || !isset($topics[$current_index])) {
            error_log("[AutoArticle Cron] Job ID {$job->id}: No valid topics found or index is out of bounds. Aborting job.");
            return ['success' => false, 'message' => 'Job failed: No valid topics found or topic index out of bounds.'];
        }

        $topic = $topics[$current_index];
        error_log("[AutoArticle Cron] Job ID {$job->id}: Selected topic: '{$topic}'");

        // Get user ID for the job
        $user_id = $job->user_id;
        
        // Prepare the API request using the new server-to-server auth
        $api = new AutoArticle_API();
        $params = array(
            'server_auth_key' => 'a_very_secret_internal_key', // The shared secret
            'user_id_for_cron' => $user_id,
            'custom' => $topic,
            'style' => $job->style,
            'image_style' => $job->image_style,
            'image_size' => $job->image_size,
            'language' => $job->language,
            'topic_usage' => $job->topic_usage,
            'custom_writing_prompt' => $job->custom_writing_prompt,
            'custom_image_prompt' => $job->custom_image_prompt
        );

        error_log("[AutoArticle Cron] Job ID {$job->id}: Sending API request with params: " . print_r($params, true));
        $response = $api->generate_article($params);
        error_log("[AutoArticle Cron] Job ID {$job->id}: Received API response: " . print_r($response, true));


        // Check API response structure more carefully
        if ($response && !is_wp_error($response) && isset($response['success']) && $response['success'] === true && isset($response['title']) && isset($response['content'])) {
            
            $job_processed_successfully = false;
            $local_image_path_to_delete = null;
            $final_message = 'Job processed, but an unknown error occurred.';

            if (isset($job->send_to_hs) && $job->send_to_hs == 1) {
                // Ensure the required functions from hubspot-helpers.php are callable
                if (!function_exists('autoarticle_get_tokens_by_hubid') || !function_exists('autoarticle_download_compress_save_image') || !function_exists('autoarticle_create_hubspot_post_direct')) {
                    error_log('AutoArticle Cron: Missing required HubSpot helper functions for job ID: ' . $job->id);
                } else {
                    $hub_id = get_user_meta($user_id, 'autoarticle_hubspot_id', true);
                    $blog_id = get_user_meta($user_id, 'autoarticle_hubspot_blog_id', true);
                    $author_id = get_user_meta($user_id, 'autoarticle_hubspot_author_id', true);

                    if (empty($hub_id) || empty($blog_id)) {
                        $error_msg = "HubSpot Portal ID or Blog ID not configured for user ID: {$user_id}.";
                        error_log("[AutoArticle Cron] Job ID {$job->id}: {$error_msg}");
                        return ['success' => false, 'message' => $error_msg];
                    } else {
                        $article_title = $response['title'];
                        $article_content = $response['content'];
                        $original_image_url = $response['image_url'] ?? '';
                        $featured_image_alt_text = $response['image_description'] ?? '';
                        
                        // Determine post status for HubSpot
                        $post_status_for_hubspot = isset($job->post_status) ? $job->post_status : get_user_meta($user_id, 'autoarticle_post_status', true);
                        if (!$post_status_for_hubspot) $post_status_for_hubspot = 'publish'; // Default
                        $publish_immediately = ($post_status_for_hubspot === 'publish');

                        $custom_slug = '';
                        if (isset($job->use_topic_as_slug)) {
                            $use_topic_as_slug = (bool)$job->use_topic_as_slug;
                            $prefix = isset($job->slug_prefix) ? $job->slug_prefix : '';
                            $suffix = isset($job->slug_suffix) ? $job->slug_suffix : '';
                            $base = $use_topic_as_slug ? $topic : $article_title;
                            
                            $full_slug_string = $prefix . $base . $suffix;

                            if ($use_topic_as_slug) {
                                $full_slug_string .= '-' . time();
                            }

                            if(!empty(trim($full_slug_string))) {
                                $custom_slug = sanitize_title($full_slug_string);
                            }
                        }

                        $final_image_url_for_hubspot = $original_image_url; // Default to original
                        $wp_processed_image_url = ''; // URL of the image processed by WP

                        if (!empty($original_image_url)) {
                            error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Original image URL from Prose API: '{$original_image_url}'");
                            $image_identifier = 'cron-job-' . $job->id . '-user-' . $user_id;

                            // Ensure autoarticle_download_compress_save_image function exists before calling
                            if (function_exists('autoarticle_download_compress_save_image')) {
                                $image_processing_result = autoarticle_download_compress_save_image($original_image_url, $image_identifier, 'AutoArticle Cron HubSpot Prep');
                                
                                $log_img_proc_url = '[Not Set]'; $log_img_proc_path = '[Not Set]';
                                if (is_array($image_processing_result)) {
                                    $log_img_proc_url = isset($image_processing_result['url']) ? $image_processing_result['url'] : '[URL Key Missing]';
                                    $log_img_proc_path = isset($image_processing_result['path']) ? $image_processing_result['path'] : '[Path Key Missing]';
                                }
                                error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Result from autoarticle_download_compress_save_image - URL: {$log_img_proc_url}, Path: {$log_img_proc_path}");

                                if ($image_processing_result && !empty($image_processing_result['url']) && !empty($image_processing_result['path'])) {
                                    $wp_processed_image_url = $image_processing_result['url'];
                                    // Ensure $local_image_path_to_delete is correctly assigned for cleanup in finally block
                                    if (empty($local_image_path_to_delete)) { // It should be null here from earlier initialization
                                        $local_image_path_to_delete = $image_processing_result['path'];
                                    }

                                    $file_name_for_hubspot = basename($image_processing_result['path']);
                                    $file_type_info = wp_check_filetype($image_processing_result['path']);
                                    $processed_file_mime_type = $file_type_info['type'];
                                    error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Determined MIME type for local image: '{$processed_file_mime_type}'");

                                    if ($processed_file_mime_type && function_exists('autoarticle_upload_image_to_hubspot')) {
                                        error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Attempting HubSpot image upload for '{$image_processing_result['path']}'.");
                                        $hubspot_upload_folder_filter_tag = 'autoarticle_hubspot_image_upload_folder'; 
                                        $hubspot_upload_folder = apply_filters($hubspot_upload_folder_filter_tag, '/autoarticle_wp_images');
                                        
                                        $upload_to_hs_result = autoarticle_upload_image_to_hubspot(
                                            $hub_id,
                                            $image_processing_result['path'], 
                                            $file_name_for_hubspot,
                                            $processed_file_mime_type,
                                            $hubspot_upload_folder
                                        );

                                        if ($upload_to_hs_result && !empty($upload_to_hs_result['success']) && !empty($upload_to_hs_result['url'])) {
                                            $final_image_url_for_hubspot = $upload_to_hs_result['url'];
                                            error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Image successfully uploaded to HubSpot: {$final_image_url_for_hubspot}");
                                        } else {
                                            $final_image_url_for_hubspot = $wp_processed_image_url; 
                                            $upload_error_message = isset($upload_to_hs_result['message']) ? $upload_to_hs_result['message'] : 'Unknown error during HubSpot image upload.';
                                            error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Failed to upload image to HubSpot. Using WP image URL. Error: {$upload_error_message}");
                                        }
                                    } elseif (!$processed_file_mime_type) {
                                        $final_image_url_for_hubspot = $wp_processed_image_url; 
                                        error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Could not determine MIME type for image: {$image_processing_result['path']}. Using WP image URL.");
                                    } else { // autoarticle_upload_image_to_hubspot function does not exist
                                         $final_image_url_for_hubspot = $wp_processed_image_url;
                                         error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: autoarticle_upload_image_to_hubspot function not found. Using WP image URL.");
                                    }
                                } else {
                                    error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Local image processing (download/compress) failed or no path returned. Original URL: {$original_image_url}. Using original image URL for HubSpot.");
                                }
                            } else { // autoarticle_download_compress_save_image function does not exist
                                error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: autoarticle_download_compress_save_image function not found. Using original image URL for HubSpot.");
                            }
                        } else {
                             error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Original image URL was empty. Skipping HubSpot image upload process.");
                        }

                        error_log("[AutoArticle Cron (Plugin)] Job ID {$job->id}: Final image URL for HubSpot post: '{$final_image_url_for_hubspot}'");
                        // Meta description can be passed if available, otherwise autoarticle_create_hubspot_post_direct will generate it
                        $meta_description = substr(strip_tags($article_content), 0, 300); // Or pass $response['meta_description'] if API provides it

                        $hubspot_result = autoarticle_create_hubspot_post_direct(
                            $hub_id,
                            $blog_id,
                            $article_title,
                            $article_content,
                            $final_image_url_for_hubspot,
                            $author_id,
                            $publish_immediately,
                            $meta_description,
                            $featured_image_alt_text,
                            $custom_slug
                        );

                        if ($hubspot_result && isset($hubspot_result['success']) && $hubspot_result['success']) {
                            $job_processed_successfully = true;
                            $final_message = 'Article successfully created in HubSpot: ' . $hubspot_result['data']['hubspot_url'];
                            error_log("[AutoArticle Cron] Job ID {$job->id}: " . $final_message);
                        } else {
                            $error_message = isset($hubspot_result['message']) ? $hubspot_result['message'] : 'Unknown error during HubSpot post creation.';
                            error_log("[AutoArticle Cron] Job ID {$job->id}: Failed to create HubSpot post. Reason: {$error_message}");
                            $final_message = "Failed to create HubSpot post: {$error_message}";
                        }
                    }
                }
            } else {
                // Original WordPress post creation logic
                $post_status = isset($job->post_status) ? $job->post_status : get_user_meta($user_id, 'autoarticle_post_status', true);
                if (!$post_status) {
                    $post_status = 'publish'; // Default fallback
                }
                
                $post_data = array(
                    'post_title' => $response['title'],
                    'post_content' => $response['content'],
                    'post_status' => $post_status,
                    'post_author' => $user_id
                );
                
                if (isset($job->use_topic_as_slug)) {
                    $use_topic_as_slug = (bool)$job->use_topic_as_slug;
                    $prefix = isset($job->slug_prefix) ? $job->slug_prefix : '';
                    $suffix = isset($job->slug_suffix) ? $job->slug_suffix : '';
                    $base = $use_topic_as_slug ? $topic : $response['title'];
                    
                    $full_slug_string = $prefix . $base . $suffix;

                    if ($use_topic_as_slug) {
                        $full_slug_string .= '-' . time();
                    }

                    if(!empty(trim($full_slug_string))) {
                        $post_data['post_name'] = sanitize_title($full_slug_string);
                    }
                }

                $post_id = wp_insert_post($post_data);
                
                if ($post_id && isset($response['image_url']) && !empty($response['image_url'])) {
                    // Handle image attachment code
                    $image_url = $response['image_url'];
                    $image_alt_text = isset($response['image_description']) ? $response['image_description'] : '';
                    $upload_dir = wp_upload_dir();
                    $filename = 'autoarticle-' . $post_id . '-' . time() . '.jpg';
                    
                    $image_data_response = wp_remote_get($image_url, array('timeout' => 30)); // Increased timeout
                    if (!is_wp_error($image_data_response) && wp_remote_retrieve_response_code($image_data_response) === 200) {
                        $image_data_body = wp_remote_retrieve_body($image_data_response);
                        $file = wp_mkdir_p($upload_dir['path']) ? $upload_dir['path'] . '/' . $filename : $upload_dir['basedir'] . '/' . $filename;
                        file_put_contents($file, $image_data_body);
                        
                        require_once(ABSPATH . 'wp-admin/includes/image.php');
                        $image = wp_get_image_editor($file);
                        if (!is_wp_error($image)) {
                            $image->set_quality(85);
                            $image->save($file);
                        }
                        
                        $wp_filetype = wp_check_filetype($filename, null);
                        $attachment_title = !empty($image_alt_text) ? $image_alt_text : sanitize_file_name($filename);
                        $attachment = array(
                            'post_mime_type' => $wp_filetype['type'],
                            'post_title' => $attachment_title, 
                            'post_content' => '',
                            'post_status' => 'inherit'
                        );
                        
                        $attach_id = wp_insert_attachment($attachment, $file, $post_id);
                        
                        if ($attach_id) {
                            $attach_data = wp_generate_attachment_metadata($attach_id, $file);
                            wp_update_attachment_metadata($attach_id, $attach_data);
                            set_post_thumbnail($post_id, $attach_id);
                            if (!empty($image_alt_text)) {
                                update_post_meta($attach_id, '_wp_attachment_image_alt', $image_alt_text);
                            }
                        }
                    } else {
                        error_log('AutoArticle Cron: Failed to download image for post ID ' . $post_id . '. URL: ' . $image_url . (is_wp_error($image_data_response) ? ' Error: ' . $image_data_response->get_error_message() : ' Status Code: ' . wp_remote_retrieve_response_code($image_data_response)));
                    }
                }
                
                if ($post_id) {
                    $job_processed_successfully = true;
                    $final_message = 'Article successfully created as WordPress post: ' . get_permalink($post_id);
                    error_log("[AutoArticle Cron] Job ID {$job->id}: " . $final_message);
                } else {
                    $final_message = 'Failed to create WordPress post.';
                    error_log("[AutoArticle Cron] Job ID {$job->id}: {$final_message}");
                }
            }
            
            if ($job_processed_successfully) {
                $next_index = ($current_index + 1) % count($topics);
                $next_run = $this->calculate_next_run($job->frequency);
                $wpdb->update($table_name, array('last_topic_index' => $next_index, 'next_run' => $next_run), array('id' => $job->id));
                error_log("[AutoArticle Cron] Job ID {$job->id}: Job updated for next run.");
            }

            if ($local_image_path_to_delete && file_exists($local_image_path_to_delete)) {
                unlink($local_image_path_to_delete);
            }

            return ['success' => $job_processed_successfully, 'message' => $final_message];

        } else {
            $error_msg = 'API response was not successful or malformed.';
            if (is_wp_error($response)) {
                $error_msg = 'WP_Error received: ' . $response->get_error_message();
            } elseif (isset($response['error'])) {
                $error_msg = 'API Error: ' . $response['error'];
            }
            error_log("[AutoArticle Cron] Job ID {$job->id}: {$error_msg} Response: " . print_r($response, true));
            return ['success' => false, 'message' => $error_msg];
        }
    }

    private function calculate_next_run($frequency) {
        // Use WP local time to match the timezone used in process_scheduled_jobs() query
        $now = current_time('mysql'); // Local time, not GMT
        switch ($frequency) {
            case 'twicedaily':
                // Calculate future time based on local timestamp (12 hours)
                return date('Y-m-d H:i:s', strtotime($now . ' +12 hours')); 
            case 'daily':
                // Calculate future time based on local timestamp
                return date('Y-m-d H:i:s', strtotime($now . ' +1 day')); 
            case 'weekly':
                return date('Y-m-d H:i:s', strtotime($now . ' +1 week'));
            case 'monthly':
                return date('Y-m-d H:i:s', strtotime($now . ' +1 month'));
            default:
                // If frequency is invalid or not set, return current local time
                return date('Y-m-d H:i:s'); 
        }
    }
} 