<?php
class AutoArticle_Generator {
    private $admin;
    private $cron;
    private $api;

    public function init() {
        $this->admin = new AutoArticle_Admin();
        $this->cron = new AutoArticle_Cron();
        $this->api = new AutoArticle_API();

        // Note: admin_menu and admin_enqueue_scripts hooks are added in AutoArticle_Admin constructor
        
        add_action('wp_ajax_autoarticle_verify_membership', array($this, 'verify_membership'));
        add_action('wp_ajax_autoarticle_generate_article', array($this, 'generate_article'));
        add_action('wp_ajax_autoarticle_schedule_article', array($this, 'schedule_article'));
        add_action('wp_ajax_autoarticle_logout', array($this, 'logout'));
        add_action('wp_ajax_autoarticle_save_topics', array($this, 'save_topics'));
        add_action('wp_ajax_autoarticle_save_membership', array($this, 'save_membership'));
        add_action('wp_ajax_autoarticle_save_writing_style', array($this, 'save_writing_style'));
        add_action('wp_ajax_autoarticle_save_image_style', array($this, 'save_image_style'));
        add_action('wp_ajax_autoarticle_save_language', array($this, 'save_language'));
        add_action('wp_ajax_autoarticle_save_image_size', array($this, 'save_image_size'));
        add_action('wp_ajax_autoarticle_save_schedule', array($this, 'save_schedule'));
        add_action('wp_ajax_autoarticle_save_topic_usage', array($this, 'save_topic_usage'));
        add_action('wp_ajax_autoarticle_save_post_status', array($this, 'save_post_status')); 
        add_action('wp_ajax_autoarticle_save_custom_prompt', array($this, 'save_custom_prompt'));
        add_action('wp_ajax_autoarticle_save_slug_settings', array($this, 'save_slug_settings'));
    }

    public function verify_membership() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $password = isset($_POST['password']) ? sanitize_text_field(wp_unslash($_POST['password'])) : '';
        $api = new AutoArticle_API();
        $response = $api->verify_membership($email, $password);

        if (!$response || !isset($response['success']) || $response['success'] != 1) {
            wp_send_json_error('Failed to verify membership');
            return;
        }

        // New: Make a separate call to get the accurate articles_remaining
        $articles_remaining_accurate = $api->get_articles_remaining($email, $password);

        $user_id = get_current_user_id();
        update_user_meta($user_id, 'autoarticle_membership_email', $email);
        update_user_meta($user_id, 'autoarticle_password', $password);
        
        wp_send_json_success(array(
            'membership_level' => $response['membership_level'],
            'membership_name' => $response['membership_name'],
            'articles_remaining' => $articles_remaining_accurate
        ));
    }

    public function generate_article() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if (!current_user_can('read')) { wp_send_json_error('Unauthorized', 403); return; }
        $user_id = get_current_user_id();

        // Sanitize article_data - each field individually
        $raw_article_data = isset($_POST['article_data']) ? (array) wp_unslash($_POST['article_data']) : null;
        if (!$raw_article_data || !isset($raw_article_data['success']) || !$raw_article_data['success']) {
            wp_send_json_error('Invalid article data');
            return;
        }

        $article_title   = isset($raw_article_data['title']) ? sanitize_text_field($raw_article_data['title']) : '';
        $article_content = isset($raw_article_data['content']) ? wp_kses_post($raw_article_data['content']) : '';
        $image_url       = isset($raw_article_data['image_url']) ? esc_url_raw($raw_article_data['image_url']) : '';
        $image_alt_text  = isset($raw_article_data['image_description']) ? sanitize_text_field($raw_article_data['image_description']) : '';

        if (empty($article_title)) {
            wp_send_json_error('Article title is required');
            return;
        }

        // Sanitize slug_data
        $raw_slug_data = isset($_POST['slug_data']) ? (array) wp_unslash($_POST['slug_data']) : null;

        $post_status = get_user_meta($user_id, 'autoarticle_post_status', true) ?: 'publish';
        $post_data = array(
            'post_title'   => $article_title,
            'post_content' => $article_content,
            'post_status'  => $post_status,
            'post_author'  => $user_id
        );

        // Custom slug generation
        if ($raw_slug_data) {
            $use_topic_as_slug = isset($raw_slug_data['use_topic_as_slug']) && $raw_slug_data['use_topic_as_slug'] != 'false';
            $prefix = isset($raw_slug_data['slug_prefix']) ? preg_replace('/[^a-zA-Z0-9_-]/', '', $raw_slug_data['slug_prefix']) : '';
            $suffix = isset($raw_slug_data['slug_suffix']) ? preg_replace('/[^a-zA-Z0-9_-]/', '', $raw_slug_data['slug_suffix']) : '';
            $base = '';

            if ($use_topic_as_slug && !empty($raw_slug_data['topic'])) {
                $base = sanitize_text_field($raw_slug_data['topic']);
            } else {
                $base = $article_title;
            }

            $full_slug_string = $prefix . $base . $suffix;

            if ($use_topic_as_slug) {
                $full_slug_string .= '-' . time();
            }
            
            if(!empty(trim($full_slug_string))) {
                $post_data['post_name'] = sanitize_title($full_slug_string);
            }
        }

        $post_id = wp_insert_post($post_data);
        if ($post_id) {
            if (!empty($image_url)) {
                $upload_dir = wp_upload_dir();
                $plugin_upload_dir = $upload_dir['basedir'] . '/autoarticle-content-generator';
                wp_mkdir_p( $plugin_upload_dir );
                $filename = 'autoarticle-' . $post_id . '-' . time() . '.jpg';
                $image_data_remote = wp_remote_get($image_url, array('timeout' => 15));
                if (!is_wp_error($image_data_remote) && wp_remote_retrieve_response_code($image_data_remote) === 200) {
                    $image_contents = wp_remote_retrieve_body($image_data_remote);
                    $file = $plugin_upload_dir . '/' . $filename;
                    file_put_contents($file, $image_contents);
                    require_once(ABSPATH . 'wp-admin/includes/image.php');
                    $image = wp_get_image_editor($file);
                    if (!is_wp_error($image)) {
                        $image->set_quality(85);
                        $image->save($file);
                    }
                    $wp_filetype = wp_check_filetype($filename, null);
                    $attachment_title = !empty($image_alt_text) ? $image_alt_text : sanitize_file_name($filename);
                    $attachment = array(
                        'post_mime_type' => $wp_filetype['type'],
                        'post_title'    => sanitize_text_field($attachment_title),
                        'post_content'  => '',
                        'post_status'   => 'inherit'
                    );
                    $attach_id = wp_insert_attachment($attachment, $file, $post_id);
                    if ($attach_id) {
                        $attach_data = wp_generate_attachment_metadata($attach_id, $file);
                        wp_update_attachment_metadata($attach_id, $attach_data);
                        set_post_thumbnail($post_id, $attach_id);
                        if (!empty($image_alt_text)) {
                            update_post_meta($attach_id, '_wp_attachment_image_alt', sanitize_text_field($image_alt_text));
                        }
                    }
                }
            }
            $email = get_user_meta($user_id, 'autoarticle_membership_email', true);
            $password = get_user_meta($user_id, 'autoarticle_password', true);
            $articles_remaining = 0;
            if ($email && $password) {
                $api = new AutoArticle_API();
                $response = $api->verify_membership($email, $password);
                if ($response && isset($response['articles_remaining'])) {
                    $articles_remaining = absint($response['articles_remaining']);
                }
            }
            wp_send_json_success(array('post_id' => $post_id, 'post_url' => get_permalink($post_id), 'articles_remaining' => $articles_remaining));
        } else {
            wp_send_json_error('Failed to create post');
        }
    }

    public function logout() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        delete_user_meta($user_id, 'autoarticle_membership_level');
        delete_user_meta($user_id, 'autoarticle_membership_email');
        wp_send_json_success();
    }

    public function save_topics() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $topics_raw = isset($_POST['topics']) ? wp_unslash($_POST['topics']) : array(); 
        
        if (!is_array($topics_raw)) {
             $topics_raw = array();
        }

        $topics = array_map(function($topic) {
            return sanitize_text_field($topic); 
        }, $topics_raw);
        
        $topics = array_filter($topics, function($topic) {
            return !empty($topic) && strlen($topic) >= 3;
        });
        
        $topics = array_values($topics);
        update_user_meta($user_id, 'autoarticle_topics', $topics);
        $this->update_scheduled_job($user_id, 'topics');
        
        wp_send_json_success(array('topics' => $topics));
    }

    public function save_membership() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $password = isset($_POST['password']) ? sanitize_text_field(wp_unslash($_POST['password'])) : '';
        
        if ($email && $password) {
            update_user_meta($user_id, 'autoarticle_membership_email', $email);
            update_user_meta($user_id, 'autoarticle_password', $password);
            wp_send_json_success();
        } else {
            wp_send_json_error('Invalid membership data');
        }
    }

    private function update_scheduled_job($user_id, $style_type = '') {
        global $wpdb;
        $table_name = $wpdb->prefix . 'autoarticle_scheduled';
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE user_id = %d",
            $user_id
        ));
        
        $update_data = array(); // Initialize update_data array

        if ($existing) {
            // $update_data = array(); // This was a redundant initialization inside the if block
            switch($style_type) {
                case 'writing':
                    $update_data['style'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'professional';
                    break;
                case 'image':
                    $update_data['image_style'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'realistic';
                    break;
                case 'language':
                    $update_data['language'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'en'; 
                    break;
                case 'image_size':
                    $update_data['image_size'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'square';
                    break;
                case 'schedule':
                    // This case is more complex and handled primarily by save_schedule(), 
                    // but if called here, ensure it updates relevant fields if an existing record.
                    $frequency = isset($_POST['frequency']) ? sanitize_text_field(wp_unslash($_POST['frequency'])) : $existing->frequency; // Keep existing if not set
                    $time = isset($_POST['schedule_time']) ? sanitize_text_field(wp_unslash($_POST['schedule_time'])) : '09:00';
                    if (property_exists($existing, 'next_run') && $existing->next_run) { // Check if next_run property exists on existing object
                         $time = gmdate('H:i', strtotime($existing->next_run)); // get current time from existing if available
                    }

                    $update_data['frequency'] = $frequency;
                    // next_run is calculated and set in save_schedule and relies on current frequency/time
                    // For a simple update here, we might just update frequency.
                    // Or, if this path is hit, it might mean only ancillary data for schedule is changing.
                    // For safety, next_run should be re-calculated if frequency/time changes.
                    // The save_schedule method is more robust for this.
                    // Let's assume if 'schedule' is passed here, it's for ancillary data or frequency has been updated by save_schedule.
                    // $update_data['next_run'] = $this->calculate_next_run($frequency, $time); // Avoid recalculating here if save_schedule handles it.
                    break;
                case 'topics':
                    $topics_json_val = $this->get_valid_topics_json($user_id);
                    $update_data['topics'] = $topics_json_val;
                    break;
                case 'topic_usage':
                    $update_data['topic_usage'] = isset($_POST['topic_usage']) ? sanitize_text_field(wp_unslash($_POST['topic_usage'])) : 'exact_match';
                    break;
                case 'post_status':
                    $update_data['post_status'] = isset($_POST['post_status']) ? sanitize_key(wp_unslash($_POST['post_status'])) : 'publish';
                    break;
                case 'send_to_hs':
                    $update_data['send_to_hs'] = isset($_POST['send_to_hs']) ? intval($_POST['send_to_hs']) : 0;
                    break;
                case 'custom_writing_prompt':
                    $update_data['custom_writing_prompt'] = isset($_POST['prompt_value']) ? sanitize_textarea_field(wp_unslash($_POST['prompt_value'])) : '';
                    break;
                case 'custom_image_prompt':
                    $update_data['custom_image_prompt'] = isset($_POST['prompt_value']) ? sanitize_textarea_field(wp_unslash($_POST['prompt_value'])) : '';
                    break;
                case 'slug_settings':
                    $update_data['use_topic_as_slug'] = isset($_POST['use_topic_as_slug']) ? 1 : 0;
                    $update_data['slug_prefix'] = isset($_POST['slug_prefix']) ? sanitize_text_field(wp_unslash($_POST['slug_prefix'])) : '';
                    $update_data['slug_suffix'] = isset($_POST['slug_suffix']) ? sanitize_text_field(wp_unslash($_POST['slug_suffix'])) : '';
                    break;
            }
            
            if (!empty($update_data)) {
                $wpdb->update(
                    $table_name,
                    $update_data,
                    array('user_id' => $user_id)
                );
            }
        } else {
            // If the row does not exist, and we are trying to update something,
            // we should create the row with default values.
            $insert_data = array(
                'user_id' => $user_id,
                'topics' => '[]',
                'style' => get_user_meta($user_id, 'autoarticle_writing_style', true) ?: 'professional',
                'image_style' => get_user_meta($user_id, 'autoarticle_image_style', true) ?: 'realistic',
                'language' => get_user_meta($user_id, 'autoarticle_language', true) ?: 'en',
                'image_size' => get_user_meta($user_id, 'autoarticle_image_size', true) ?: 'square',
                'frequency' => '', 
                'next_run' => null, 
                'topic_usage' => get_user_meta($user_id, 'autoarticle_topic_usage', true) ?: 'exact_match',
                'post_status' => get_user_meta($user_id, 'autoarticle_post_status', true) ?: 'publish',
                'last_topic_index' => 0,
                'send_to_hs' => 0,
                'custom_writing_prompt' => '',
                'custom_image_prompt' => '',
                'use_topic_as_slug' => 0,
                'slug_prefix' => '',
                'slug_suffix' => ''
            );

            // Apply the current change if it's one of these fields
            // This ensures the specific field being updated is correctly set in the new row.
            switch($style_type) {
                case 'topics':
                    $insert_data['topics'] = $this->get_valid_topics_json($user_id);
                    break;
                case 'writing':
                    $insert_data['style'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : $insert_data['style'];
                    break;
                case 'image':
                    $insert_data['image_style'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : $insert_data['image_style'];
                    break;
                case 'language':
                    $insert_data['language'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : $insert_data['language'];
                    break;
                case 'image_size':
                    $insert_data['image_size'] = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : $insert_data['image_size'];
                    break;
                case 'topic_usage':
                    $insert_data['topic_usage'] = isset($_POST['topic_usage']) ? sanitize_text_field(wp_unslash($_POST['topic_usage'])) : $insert_data['topic_usage'];
                    break;
                case 'post_status':
                    $insert_data['post_status'] = isset($_POST['post_status']) ? sanitize_key(wp_unslash($_POST['post_status'])) : $insert_data['post_status'];
                    break;
                case 'send_to_hs':
                    $insert_data['send_to_hs'] = isset($_POST['send_to_hs']) ? intval($_POST['send_to_hs']) : $insert_data['send_to_hs'];
                    break;
                case 'custom_writing_prompt':
                    $insert_data['custom_writing_prompt'] = isset($_POST['prompt_value']) ? sanitize_textarea_field(wp_unslash($_POST['prompt_value'])) : '';
                    break;
                case 'custom_image_prompt':
                    $insert_data['custom_image_prompt'] = isset($_POST['prompt_value']) ? sanitize_textarea_field(wp_unslash($_POST['prompt_value'])) : '';
                    break;
                case 'slug_settings':
                    $insert_data['use_topic_as_slug'] = isset($_POST['use_topic_as_slug']) ? 1 : 0;
                    $insert_data['slug_prefix'] = isset($_POST['slug_prefix']) ? sanitize_text_field(wp_unslash($_POST['slug_prefix'])) : '';
                    $insert_data['slug_suffix'] = isset($_POST['slug_suffix']) ? sanitize_text_field(wp_unslash($_POST['slug_suffix'])) : '';
                    break;
                // 'schedule' type has its own comprehensive insert logic in save_schedule(),
                // so we don't attempt to insert for 'schedule' here.
            }

            // Only insert if the call was for a type that should create a row.
            // This specifically ensures 'topics', 'writing', 'image', etc., can create the row if it's missing.
            // Avoid inserting for 'schedule' as 'save_schedule' handles its own insertions.
            if ($style_type !== 'schedule' && $style_type !== '') { // Also ensure style_type is not empty
                 $wpdb->insert($table_name, $insert_data);
            }
        }
    }

    public function save_writing_style() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $style = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'professional';
        update_user_meta($user_id, 'autoarticle_writing_style', $style);
        $this->update_scheduled_job($user_id, 'writing');
        wp_send_json_success();
    }

    public function save_image_style() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $style = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'realistic';
        update_user_meta($user_id, 'autoarticle_image_style', $style);
        $this->update_scheduled_job($user_id, 'image');
        wp_send_json_success();
    }

    public function save_language() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $language = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'en'; 
        update_user_meta($user_id, 'autoarticle_language', $language);
        $this->update_scheduled_job($user_id, 'language');
        wp_send_json_success();
    }

    public function save_image_size() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $size = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : 'square';
        update_user_meta($user_id, 'autoarticle_image_size', $size);
        $this->update_scheduled_job($user_id, 'image_size');
        wp_send_json_success();
    }

    public function save_custom_prompt() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $prompt_type = isset($_POST['prompt_type']) ? sanitize_key( wp_unslash( $_POST['prompt_type'] ) ) : '';
        $prompt_value = isset($_POST['prompt_value']) ? sanitize_textarea_field(wp_unslash($_POST['prompt_value'])) : '';

        if (in_array($prompt_type, ['custom_writing_prompt', 'custom_image_prompt'])) {
            update_user_meta($user_id, $prompt_type, $prompt_value);
            $this->update_scheduled_job($user_id, $prompt_type);
            wp_send_json_success();
        } else {
            wp_send_json_error('Invalid prompt type.');
        }
    }

    public function save_schedule() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $frequency = isset($_POST['frequency']) ? sanitize_text_field(wp_unslash($_POST['frequency'])) : '';
        $time = isset($_POST['schedule_time']) ? sanitize_text_field(wp_unslash($_POST['schedule_time'])) : '09:00';
        
        // Slug settings
        $use_topic_as_slug = isset($_POST['use_topic_as_slug']) ? 1 : 0;
        $slug_prefix = isset($_POST['slug_prefix']) ? preg_replace('/[^a-zA-Z0-9_-]/', '', wp_unslash($_POST['slug_prefix'])) : '';
        $slug_suffix = isset($_POST['slug_suffix']) ? preg_replace('/[^a-zA-Z0-9_-]/', '', wp_unslash($_POST['slug_suffix'])) : '';

        // Determine send_to_hs based on presence and value of frequency, ONLY if from frontend shortcode
        $send_to_hs_value_to_set = null; // Initialize to null, meaning no change unless conditions are met

        $source = isset($_POST['source']) ? sanitize_key( wp_unslash( $_POST['source'] ) ) : '';
        if ($source === 'frontend_shortcode') {
            // Only if the call is from the frontend shortcode, consider changing send_to_hs
            if (array_key_exists('frequency', $_POST)) {
                // Frequency was part of the submission from the frontend
                if (!empty($frequency) && in_array($frequency, ['twicedaily', 'daily', 'weekly', 'monthly'])) {
                    $send_to_hs_value_to_set = 1;
                } else {
                    $send_to_hs_value_to_set = 0; // Set to 0 if frequency is empty or invalid from frontend
                }
            }
            // If 'frequency' is not in _POST even from frontend (e.g. only time changed via a more granular AJAX),
            // $send_to_hs_value_to_set remains null, preserving DB value.
        }
        // If the call is from the admin dashboard (no 'source' or different 'source'), 
        // $send_to_hs_value_to_set will remain null, thus preserving the DB value of send_to_hs.

        // Ensure time is valid format
        if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $time)) {
             $time = '09:00'; // Default if invalid
        }

        update_user_meta($user_id, 'autoarticle_frequency', $frequency);
        update_user_meta($user_id, 'autoarticle_schedule_time', $time);
        
        // Save slug settings to user meta
        update_user_meta($user_id, 'autoarticle_use_topic_as_slug', $use_topic_as_slug);
        update_user_meta($user_id, 'autoarticle_slug_prefix', $slug_prefix);
        update_user_meta($user_id, 'autoarticle_slug_suffix', $slug_suffix);

        global $wpdb;
        $table_name = $wpdb->prefix . 'autoarticle_scheduled';
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE user_id = %d",
            $user_id
        ));
        
        // Calculate next_run for this user's schedule (stored in DB, not WP-Cron)
        // WP-Cron runs hourly to check all scheduled jobs - see autoarticle_ensure_cron_scheduled()
        $next_run_datetime_str = null;
        if (!empty($frequency)) {
            $next_run_datetime_str = $this->calculate_next_run($frequency, $time);
        }

        if ($existing) {
            // Update existing row in DB
            $update_data = array(
                'frequency' => $frequency, // Always update frequency if submitted (even if empty to clear it)
                'next_run' => $next_run_datetime_str, 
                 'style' => get_user_meta($user_id, 'autoarticle_writing_style', true) ?: 'professional',
                 'image_style' => get_user_meta($user_id, 'autoarticle_image_style', true) ?: 'realistic',
                 'language' => get_user_meta($user_id, 'autoarticle_language', true) ?: 'en',
                 'image_size' => get_user_meta($user_id, 'autoarticle_image_size', true) ?: 'square',
                 'topic_usage' => get_user_meta($user_id, 'autoarticle_topic_usage', true) ?: 'exact_match',
                 'post_status' => get_user_meta($user_id, 'autoarticle_post_status', true) ?: 'publish',
                 'topics' => $this->get_valid_topics_json($user_id), 
                 'custom_writing_prompt' => get_user_meta($user_id, 'custom_writing_prompt', true),
                 'custom_image_prompt' => get_user_meta($user_id, 'custom_image_prompt', true),
                 'use_topic_as_slug' => $use_topic_as_slug,
                 'slug_prefix' => $slug_prefix,
                 'slug_suffix' => $slug_suffix
            );

            // Only include send_to_hs in the update if frequency was explicitly submitted
            if ($send_to_hs_value_to_set !== null) {
                $update_data['send_to_hs'] = $send_to_hs_value_to_set;
            }
            // If only time was submitted, $send_to_hs_value_to_set will be null, 
            // and send_to_hs will not be part of the $update_data, preserving its existing DB value.

            $wpdb->update(
                $table_name,
                $update_data,
                array('user_id' => $user_id)
            );
             // Delete row from DB if frequency is now empty
             if (empty($frequency)) {
                 $wpdb->delete($table_name, array('user_id' => $user_id));
             }

        } elseif (!empty($frequency) && $next_run_datetime_str) { // Only insert if frequency is set AND next_run was calculated
            // Insert new row in DB
            $wpdb->insert(
                $table_name,
                array(
                    'user_id' => $user_id,
                    'topics' => $this->get_valid_topics_json($user_id), 
                    'style' => get_user_meta($user_id, 'autoarticle_writing_style', true) ?: 'professional',
                    'image_style' => get_user_meta($user_id, 'autoarticle_image_style', true) ?: 'realistic',
                    'language' => get_user_meta($user_id, 'autoarticle_language', true) ?: 'en',
                    'image_size' => get_user_meta($user_id, 'autoarticle_image_size', true) ?: 'square',
                    'frequency' => $frequency,
                    'next_run' => $next_run_datetime_str, 
                    'topic_usage' => get_user_meta($user_id, 'autoarticle_topic_usage', true) ?: 'exact_match',
                    'post_status' => get_user_meta($user_id, 'autoarticle_post_status', true) ?: 'publish',
                    'last_topic_index' => 0, 
                    'send_to_hs' => ($send_to_hs_value_to_set !== null) ? $send_to_hs_value_to_set : 0,
                    'custom_writing_prompt' => get_user_meta($user_id, 'custom_writing_prompt', true),
                    'custom_image_prompt' => get_user_meta($user_id, 'custom_image_prompt', true),
                    'use_topic_as_slug' => $use_topic_as_slug,
                    'slug_prefix' => $slug_prefix,
                    'slug_suffix' => $slug_suffix
                )
            );
        } 
        
        wp_send_json_success(array(
            'next_run' => $next_run_datetime_str // Send back the calculated next_run time string (or null)
        ));
    }
    
    // Helper function to avoid repetition - adjust as needed
    private function get_valid_topics_json($user_id) {
        $topics_json = get_user_meta($user_id, 'autoarticle_topics', true);
        if (empty($topics_json)) {
             $topics_json = '[]';
        } else if (is_array($topics_json)) {
             $topics_json = json_encode($topics_json);
        }
        json_decode($topics_json);
        if (json_last_error() !== JSON_ERROR_NONE) {
             $topics_json = '[]';
        }
        return $topics_json;
    }

    private function calculate_next_run($frequency, $time) {
        // If frequency is empty/disabled, don't calculate a next run time.
        if (empty($frequency)) {
            // Return a value indicating it's disabled, perhaps null or a very old date.
            // Let's return null and handle it in save_schedule.
            return null; 
        }

        $time_parts = explode(':', $time);
        if (count($time_parts) !== 2) {
             // Default to 9:00 if time format is invalid
             $hour = 9;
             $minute = 0;
        } else {
            $hour = intval($time_parts[0]);
            $minute = intval($time_parts[1]);
        }

        // Get WordPress timezone
        $timezone_string = wp_timezone_string();
        $timezone = new DateTimeZone($timezone_string);
        
        // Get current time in WP timezone
        $now_dt = new DateTime('now', $timezone);
        
        // Create DateTime object for the target time today
        $next_run_dt = new DateTime('now', $timezone);
        $next_run_dt->setTime($hour, $minute, 0); // Set time, reset seconds to 0

        // If the calculated time is in the past today, move to the next interval
        if ($next_run_dt <= $now_dt) {
            switch($frequency) {
                case 'twicedaily':
                    $next_run_dt->modify('+12 hours');
                    break;
                case 'daily':
                    $next_run_dt->modify('+1 day');
                    break;
                case 'weekly':
                     // If today is already past the target time, find the next occurrence
                     // This ensures it lands on the same day of the week next week
                     $next_run_dt->modify('+1 week'); 
                    break;
                case 'monthly':
                    // If today is already past the target time, find the next occurrence next month
                    $next_run_dt->modify('+1 month');
                    break;
                default:
                    // If frequency is invalid but not empty, default to next day
                     $next_run_dt->modify('+1 day');
                    break;
            }
        } else {
             // If the time is later today, check frequency for weekly/monthly initial offset
             switch($frequency) {
                 case 'weekly':
                      // If it's weekly and the time is later today, it should still be today unless modified above
                      // No change needed here unless we specifically wanted to force it to next week
                      break;
                 case 'monthly':
                      // If it's monthly and the time is later today, it should still be today unless modified above
                      // No change needed here unless we specifically wanted to force it to next month
                      break;
                 case 'daily':
                 default:
                      // Time is later today, keep it as is for daily or default
                      break;
             }
        }

        // Format for database storage (MySQL DATETIME format in UTC)
        // Although current_time('mysql') uses WP timezone, WP Cron usually works best with UTC timestamps.
        // Let's store in WP timezone format 'Y-m-d H:i:s' as used by current_time('mysql') and the cron check.
        return $next_run_dt->format('Y-m-d H:i:s');
    }

    public function save_slug_settings() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();

        $use_topic_as_slug = isset($_POST['use_topic_as_slug']) ? 1 : 0;
        $slug_prefix = isset($_POST['slug_prefix']) ? preg_replace('/[^a-zA-Z0-9_-]/', '', wp_unslash($_POST['slug_prefix'])) : '';
        $slug_suffix = isset($_POST['slug_suffix']) ? preg_replace('/[^a-zA-Z0-9_-]/', '', wp_unslash($_POST['slug_suffix'])) : '';
        
        update_user_meta($user_id, 'autoarticle_use_topic_as_slug', $use_topic_as_slug);
        update_user_meta($user_id, 'autoarticle_slug_prefix', $slug_prefix);
        update_user_meta($user_id, 'autoarticle_slug_suffix', $slug_suffix);

        $this->update_scheduled_job($user_id, 'slug_settings');
        
        wp_send_json_success();
    }

    public function get_remaining_articles($user_id) {
        $email = get_user_meta($user_id, 'autoarticle_membership_email', true);
        $password = get_user_meta($user_id, 'autoarticle_password', true);
        
        if (!$email || !$password) {
            return 0;
        }
        
        $api = new AutoArticle_API();
        return $api->get_articles_remaining($email, $password);
    }

    public function save_topic_usage() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $topic_usage = isset($_POST['topic_usage']) ? sanitize_text_field(wp_unslash($_POST['topic_usage'])) : 'exact_match';
        
        // Validate topic_usage value
        if (!in_array($topic_usage, array('exact_match', 'relevant_articles'))) {
            $topic_usage = 'exact_match';
        }
        
        update_user_meta($user_id, 'autoarticle_topic_usage', $topic_usage);
        $this->update_scheduled_job($user_id, 'topic_usage');
        
        wp_send_json_success();
    }

    public function save_post_status() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        $post_status = isset($_POST['post_status']) ? sanitize_key(wp_unslash($_POST['post_status'])) : 'publish';
        
        // Additional validation if needed, e.g., check against allowed post statuses
        $allowed_statuses = get_post_stati();
        if (!array_key_exists($post_status, $allowed_statuses)) {
             $post_status = 'publish'; // Default to publish if invalid
        }

        update_user_meta($user_id, 'autoarticle_post_status', $post_status);
        $this->update_scheduled_job($user_id, 'post_status');

        wp_send_json_success();
    }

    public function schedule_article() {
        check_ajax_referer('autoarticle_nonce', 'nonce');
        if ( ! current_user_can( 'read' ) ) {
            wp_send_json_error( 'Unauthorized', 403 );
            return;
        }
        $user_id = get_current_user_id();
        // Retrieve settings from user meta first
        $topics = $this->get_valid_topics_json($user_id); // Use helper to get valid JSON string
        $style = get_user_meta($user_id, 'autoarticle_writing_style', true) ?: 'professional';
        $image_style = get_user_meta($user_id, 'autoarticle_image_style', true) ?: 'realistic';
        $language = get_user_meta($user_id, 'autoarticle_language', true) ?: 'en';
        $image_size = get_user_meta($user_id, 'autoarticle_image_size', true) ?: 'square';
        $topic_usage = get_user_meta($user_id, 'autoarticle_topic_usage', true) ?: 'exact_match';
        $post_status = get_user_meta($user_id, 'autoarticle_post_status', true) ?: 'publish';

        // Get frequency and time from POST, sanitize and unslash
        $frequency = isset($_POST['frequency']) ? sanitize_text_field(wp_unslash($_POST['frequency'])) : '';
        $schedule_time = isset($_POST['schedule_time']) ? sanitize_text_field(wp_unslash($_POST['schedule_time'])) : '09:00';
        $send_to_hs = isset($_POST['send_to_hs']) ? intval($_POST['send_to_hs']) : 0;

        // Basic validation
        if (empty($topics) || $topics === '[]') { // Check if topics are empty
            wp_send_json_error('Topics cannot be empty.');
            return;
        }
        // Validate schedule time format
        if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $schedule_time)) {
             $schedule_time = '09:00'; // Default if invalid
        }
        
        // Calculate next run time (will return null if frequency is empty)
        $next_run = $this->calculate_next_run($frequency, $schedule_time); 

        global $wpdb;
        $table_name = $wpdb->prefix . 'autoarticle_scheduled';
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE user_id = %d",
            $user_id
        ));

        $data = array(
            'user_id'       => $user_id,
            'topics'        => $topics, // Already validated JSON string
            'style'         => $style,
            'image_style'   => $image_style,
            'language'      => $language,
            'image_size'    => $image_size,
            'frequency'     => $frequency,
            'next_run'      => $next_run, // Can be null if frequency is empty
            'topic_usage'   => $topic_usage,
            'post_status'   => $post_status,
            // Reset index only when inserting or if topics changed?
            // For simplicity, let's always update/set index to 0 here
            'last_topic_index' => 0, 
            'send_to_hs' => $send_to_hs // Added send_to_hs
        );

        if ($existing) {
            if (empty($frequency)) {
                 // If frequency is disabled, delete the schedule
                 $wpdb->delete($table_name, array('user_id' => $user_id));
                 $next_run = null; // Ensure response reflects deletion
            } else {
                // Update existing schedule
                $wpdb->update(
                    $table_name,
                    $data,
                    array('user_id' => $user_id)
                );
            }
        } elseif (!empty($frequency)) { // Only insert if frequency is not empty
            // Insert new schedule
            $wpdb->insert($table_name, $data);
        } else {
             // Frequency is empty and no existing schedule, do nothing.
             $next_run = null;
        }
        
        // Note: WP-Cron is scheduled globally via autoarticle_ensure_cron_scheduled()
        // It runs hourly and checks the DB for any due jobs

        wp_send_json_success(array(
            'next_run' => $next_run,
            // Format using gmdate, check if $next_run is null before formatting
            'formatted_next_run' => $next_run ? gmdate('F j, Y, g:i a', strtotime($next_run)) : 'Not scheduled' 
        ));
    }
} 